import {
  ChangeDetectionStrategy,
  Component,
  HostBinding,
  Input,
} from "@angular/core";
import {
  DomSanitizer,
  SafeResourceUrl,
} from "@angular/platform-browser";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {User} from "@models";
import {isNullOrUndefined} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-avatar",
  templateUrl: "./shared-avatar.presentational.html",
  styleUrls: ["./shared-avatar.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAvatarPresentational extends SharedAbstractPresentational {
  private _user: User;

  @Input()
  set user(value: User) {
    this._user = value;
    this.computeInitial();
  }

  get user(): User {
    return this._user;
  }

  @HostBinding("class.is-photo")
  private _photoUserSanitize: SafeResourceUrl | undefined;

  @Input()
  set photoUser(value: string) {
    if (isNullOrUndefined(value)) {
      this._photoUserSanitize = undefined;
      return;
    }
    this._photoUserSanitize = this.sanitizeUrl(value);
  }

  get photoUserSanitize(): SafeResourceUrl {
    return this._photoUserSanitize;
  }

  private readonly DEFAULT_INITIAL: string = "";

  initial: string = this.DEFAULT_INITIAL;

  constructor(private readonly _sanitizer: DomSanitizer) {
    super();
  }

  computeInitial(): void {
    if (isNullOrUndefined(this._user)) {
      this.initial = this.DEFAULT_INITIAL;
    } else {
      this.initial = this._user.firstName.substring(0, 1) + this._user.lastName.substring(0, 1);
    }
  }

  sanitizeUrl(url: string | undefined): undefined | SafeResourceUrl {
    if (isNullOrUndefined(url)) {
      return undefined;
    } else {
      return this._sanitizer.bypassSecurityTrustResourceUrl(url);
    }
  }
}
