import {
  ChangeDetectionStrategy,
  Component,
  HostBinding,
  HostListener,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {environment} from "@environments/environment";
import {Archive} from "@home/models/archive.model";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {MatButtonThemeClassName} from "@shared/directives/shared-mat-button-theme/shared-mat-button-theme.directive";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {MetadataUtil} from "@shared/utils/metadata.util";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  DateUtil,
  isNullOrUndefined,
  ObservableUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-archive-tile",
  templateUrl: "./shared-archive-tile.presentational.html",
  styleUrls: ["./shared-archive-tile.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedArchiveTilePresentational extends SharedAbstractPresentational implements OnInit {
  @Input()
  archive: Archive;

  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnInput;

  _mode: SharedArchiveTileMode;

  @Input()
  set mode(value: SharedArchiveTileMode) {
    this._mode = value;
  }

  get mode(): SharedArchiveTileMode {
    return this._mode;
  }

  @Input()
  isNotLogged: boolean = false;

  @Input()
  alreadyInCart: boolean = false;

  @HostBinding("class.is-access-controlled")
  @Input()
  isAccessControlled: boolean = false;

  @HostBinding("class.is-line-display-mode")
  @Input()
  isLineDisplayMode: boolean = false;

  private readonly _downloadBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("downloadChange")
  readonly downloadObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._downloadBS);

  private readonly _deleteBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("deleteChange")
  readonly deleteObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._deleteBS);

  private readonly _addToCartBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("addToCartChange")
  readonly addToCartObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._addToCartBS);

  private readonly _requestAccessBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("requestAccessChange")
  readonly requestAccessObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._requestAccessBS);

  private readonly _selectBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("selectChange")
  readonly selectObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._selectBS);

  @HostListener("click", ["$event"]) click(mouseEvent: MouseEvent): void {
    this._selectBS.next(this.archive);
  }

  imageArchive: string | undefined;

  get publicationDate(): string {
    const publicationDate = MetadataUtil.getIssuedDate(this.archive?.archiveMetadata?.metadata);
    if (isNullOrUndefined(publicationDate)) {
      return this.archive.yearPublicationDate;
    }
    return DateUtil.convertDateToDateString(publicationDate);
  }

  get downloadButtonTheme(): MatButtonThemeClassName {
    if (this._mode === "home-tiles") {
      return "flat-button";
    }
    return "button";
  }

  constructor(readonly archiveAccessRightService: ArchiveAccessRightService) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.imageArchive = `${AccessResourceApiEnum.publicMetadata}/${this.archive.resId}/${ApiActionEnum.THUMBNAIL}`;
  }

  download(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    this._downloadBS.next(this.archive);
  }

  sendRequest(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    this._requestAccessBS.next(this.archive);
  }

  delete(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    this._deleteBS.next(this.archive);
  }

  addToCart(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    this._addToCartBS.next(this.archive);
  }
}

export type SharedArchiveTileMode = "home-tiles" | "order-draft" | "order";
export const SharedArchiveTileMode = {
  HOME_TILES: "home-tiles" as SharedArchiveTileMode,
  ORDER_DRAFT: "order-draft" as SharedArchiveTileMode,
  ORDER: "order" as SharedArchiveTileMode,
};
