import {
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
} from "@angular/core";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {Enums} from "@enums";
import {BaseResourceApiEnum} from "@shared/enums/api.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {AipCopyList} from "@shared/models/business/aip-copy-list.model";
import {AipCopy} from "@shared/models/business/aip-copy.model";
import {
  EnumUtil,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  urlSeparator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-status-summary",
  templateUrl: "./shared-aip-status-summary.presentational.html",
  styleUrls: ["./shared-aip-status-summary.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipStatusSummaryPresentational extends SharedAbstractPresentational implements OnInit {
  private _aipCopyList: AipCopyList;

  @Input()
  set aipCopyList(value: AipCopyList) {
    this._aipCopyList = value;
    this.aipCopySorted = [];
    BaseResourceApiEnum.archivalStorageList.forEach(storagion => {
      this.aipCopyList.copies.forEach(copy => {
        if (copy.storageUrl === storagion.url) {
          this.aipCopySorted.push(copy);
        }
      });
    });
  }

  get aipCopyList(): AipCopyList {
    return this._aipCopyList;
  }

  aipCopySorted: AipCopy[] = [];

  private readonly PATH_TO_AIP_STORAGION: string = "#" + urlSeparator + AppRoutesEnum.preservationPlanning + urlSeparator + PreservationPlanningRoutesEnum.aip + urlSeparator;

  get packageStatusEnum(): typeof Enums.Package.StatusEnum {
    return Enums.Package.StatusEnum;
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  getStatusLabel(status: Enums.Package.StatusEnum): string {
    if (isNullOrUndefined(status)) {
      return MARK_AS_TRANSLATABLE("COPY_MISSING");
    }
    if (status === Enums.Package.StatusEnum.IN_PREPARATION) {
      return MARK_AS_TRANSLATABLE("IN_PREPARATION");
    }
    if (status === Enums.Package.StatusEnum.REINDEXING) {
      return MARK_AS_TRANSLATABLE("REINDEXING");
    }
    if (isNullOrUndefined(status)) {
      return MARK_AS_TRANSLATABLE("UNDEFINED");
    }
    return EnumUtil.getLabel(Enums.Package.StatusEnumTranslate, status);
  }

  getStoragionNumber(aipCopy: AipCopy): number {
    const storagion = BaseResourceApiEnum.archivalStorageList.find(sto => aipCopy.storageUrl === sto.url).index;
    if (isNotNullNorUndefined(storagion)) {
      return storagion;
    }
    return 1;
  }

  getUrl(aipCopy: AipCopy): string | undefined {
    if (isNullOrUndefined(aipCopy.status)) {
      return undefined;
    }

    return this.PATH_TO_AIP_STORAGION + this.getStoragionNumber(aipCopy) + urlSeparator + SharedAipRoutesEnum.aipDetail + urlSeparator + aipCopy.aip.resId;
  }
}
