import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
} from "@angular/core";
import {
  ChangeInfo,
  User,
} from "@models";
import {Store} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {CLASS_IGNORE_EDIT_FIELD} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {SharedUserAction} from "@shared/stores/user/shared-user.action";
import {SharedUserState} from "@shared/stores/user/shared-user.state";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResource,
  BaseResourceType,
  DateUtil,
  isNonEmptyArray,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MemoizedUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-additional-information-panel-container",
  templateUrl: "./shared-additional-information-panel.container.html",
  styleUrls: ["./shared-additional-information-panel.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAdditionalInformationPanelContainer<TResourceModel extends BaseResource> extends SharedAbstractContainer {
  creationInfo: ChangeInfoDisplay | undefined;
  updateInfo: ChangeInfoDisplay | undefined;

  userListObs: Observable<User[]> = MemoizedUtil.list(this._store, SharedUserState);

  readonly classIgnoreEditField: string = CLASS_IGNORE_EDIT_FIELD;

  @Input()
  set resource(value: BaseResourceType) {
    this._init();
    if (isNullOrUndefined(value)) {
      return;
    }
    const creationInfos: ChangeInfo = value["creation"];
    this.creationInfo = this._generateChangeInfoDisplay(creationInfos);
    const lastUpdatesInfos: ChangeInfo = value["lastUpdate"];
    this.updateInfo = this._generateChangeInfoDisplay(lastUpdatesInfos);
  }

  constructor(private readonly _store: Store,
              private readonly _changeDetector: ChangeDetectorRef) {
    super();
  }

  private _init(): void {
    this.creationInfo = undefined;
    this.updateInfo = undefined;
  }

  private _generateChangeInfoDisplay(changeInfoType: ChangeInfo): ChangeInfoDisplay {
    const changeInfosDisplay = {} as ChangeInfoDisplay;
    if (isNotNullNorUndefined(changeInfoType)) {
      if (isNotNullNorUndefined(changeInfoType.when)) {
        changeInfosDisplay.date = DateUtil.convertDateToDateTimeString(new Date(changeInfoType.when));
      }
      if (isNotNullNorUndefined(changeInfoType.who)) {
        this.subscribe(this.userListObs.pipe(
          distinctUntilChanged(),
          filter(userList => isNonEmptyArray(userList) && isNotNullNorUndefined(userList.find(u => u.externalUid === changeInfoType.who))),
          take(1),
          tap(userList => {
            changeInfosDisplay.user = userList.find(u => u.externalUid === changeInfoType.who);
            this._changeDetector.detectChanges();
          }),
        ));
        this._store.dispatch(new SharedUserAction.GetByExternalUid(changeInfoType.who, true));
      }
    }

    return changeInfosDisplay;
  }
}

interface ChangeInfoDisplay {
  user?: User | undefined;
  date?: string | undefined;
}
