import {
  FlexibleConnectedPositionStrategy,
  Overlay,
  OverlayRef,
} from "@angular/cdk/overlay";
import {ComponentPortal} from "@angular/cdk/portal";
import {
  AfterViewInit,
  ChangeDetectorRef,
  ComponentRef,
  Directive,
  ElementRef,
  Input,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
  ViewContainerRef,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormBuilder,
  FormControl,
} from "@angular/forms";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {environment} from "@environments/environment";
import {SharedAbstractContentPresentational} from "@shared/components/presentationals/shared-abstract-content/shared-abstract-content.presentational";
import {SharedMultiSelectContentPresentational} from "@shared/components/presentationals/shared-multi-select-content/shared-multi-select-content.presentational";
import {SharedMultiSelectDefaultValueContentPresentational} from "@shared/components/presentationals/shared-multi-select-default-value-content/shared-multi-select-default-value-content.presentational";
import {SharedSearchableMultiSelectContentPresentational} from "@shared/components/presentationals/shared-searchable-multi-select-content/shared-searchable-multi-select-content.presentational";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {TabulationService} from "@shared/services/tabulation.service";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {tap} from "rxjs/operators";
import {
  BaseResourceType,
  ClipboardUtil,
  FormValidationHelper,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ObservableUtil,
  PropertyName,
  StringUtil,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractMultiSelectPresentational<TResource extends BaseResourceType> extends SharedAbstractPresentational implements ControlValueAccessor, OnInit, AfterViewInit, OnDestroy {

  protected _overlayRefs: Set<OverlayRef> = new Set<OverlayRef>();

  protected _elementToReFocus: Element;

  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  @Input()
  title: string;

  @Input()
  placeholder: string;

  @Input()
  valueKey: string;

  @Input()
  labelKey: string;

  @Input()
  labelCallback: (value: TResource) => string = (value => value[this.labelKey]);

  @Input()
  extraInfoLabelKey: string;

  @Input()
  extraInfoDisplayOnContent: boolean = true;

  @Input()
  extraInfoImage: IconNameEnum;

  @Input()
  formControl: FormControl;

  @Input()
  required: boolean;

  @Input()
  closeAfterValueSelected: boolean = false;

  @Input()
  hideItemSelected: boolean = false;

  @Input()
  isWithLink: boolean = false;

  @Input()
  tooltipNavigateToTranslate: string;

  selectedItems: TResource[] = [];
  removable: boolean = true;
  classInputHide: string = environment.classInputHide;

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  get isOverlayOpen(): boolean {
    return this._overlayRefs?.size > 0;
  }

  @ViewChild("inputElementRef")
  readonly input: ElementRef;

  protected readonly _valueBS: BehaviorSubject<TResource[] | undefined> = new BehaviorSubject<TResource[] | undefined>(undefined);
  @Output("valueChange")
  readonly valueObs: Observable<TResource[] | undefined> = ObservableUtil.asObservable(this._valueBS);

  protected readonly _navigateBS: BehaviorSubject<TResource | undefined> = new BehaviorSubject<TResource | undefined>(undefined);
  @Output("navigate")
  readonly navigateObs: Observable<TResource | undefined> = ObservableUtil.asObservable(this._navigateBS);

  constructor(protected _fb: FormBuilder,
              protected _elementRef: ElementRef,
              protected _viewContainerRef: ViewContainerRef,
              protected _overlay: Overlay,
              protected _changeDetector: ChangeDetectorRef,
              protected _notificationService: NotificationService,
              protected _tabulationService: TabulationService) {
    super();
  }

  ngOnInit(): void {
    this._manageInitExistingValue();
  }

  ngOnDestroy(): void {
    this.closeOverlays();
    super.ngOnDestroy();
  }

  protected abstract _manageInitExistingValue(): void;

  propagateChange = (__: any) => {};

  registerOnChange(fn: any): void {
    this.propagateChange = fn;
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState(isDisabled: boolean): void {
  }

  writeValue(value: string[]): void {
    this.orderSelectedItemByValue(value);
  }

  private orderSelectedItemByValue(value: string[]): void {
    this.selectedItems = this.selectedItems.sort((a, b) => value.indexOf(a.resId) - value.indexOf(b.resId));
  }

  abstract getComponentPortal(): any
    | ComponentPortal<SharedSearchableMultiSelectContentPresentational<TResource>>
    | ComponentPortal<SharedMultiSelectDefaultValueContentPresentational<TResource>>
    | ComponentPortal<SharedMultiSelectContentPresentational<TResource>>;

  openOverlay(inputElementRef: Element): void {
    this._elementToReFocus = inputElementRef;
    const minWidthField = this._elementRef.nativeElement.getBoundingClientRect().width;
    const overlayRef = this._overlay.create({
      positionStrategy: this._getPosition(this._elementRef),
      scrollStrategy: this._overlay.scrollStrategies.block(),
      hasBackdrop: true,
      backdropClass: "cdk-overlay-transparent-backdrop",
      width: minWidthField,
    });

    this._overlayRefs.add(overlayRef);

    const overlayContentComponent = this.getComponentPortal();
    const componentRef = overlayRef.attach(overlayContentComponent);
    componentRef.instance.host = this;

    this._observeValueSelectedInOverlay(componentRef, overlayRef);
    this._observeCloseEventInOverlay(componentRef, overlayRef);
    this._observeClickBackdrop(componentRef, overlayRef);
  }

  protected _observeValueSelectedInOverlay(componentRef: ComponentRef<SharedMultiSelectContentPresentational<TResource>>
    | ComponentRef<SharedSearchableMultiSelectContentPresentational<TResource>>,
                                           overlayRef: OverlayRef): void {
    this.subscribe(componentRef.instance.valueAddedObs.pipe(
      tap(value => {
        this.add(value);
        this.extraBehaviorOnValueSelectedInOvertly(value);
        this.formControl.markAsTouched();
        if (this.closeAfterValueSelected) {
          this._closeOverlay(overlayRef);
          this._tabulationService.focusNext(this._elementToReFocus);
        } else {
          overlayRef.updatePosition();
        }
      }),
    ));

    this.subscribe(componentRef.instance.valueRemovedObs.pipe(
      tap(value => {
        this.remove(value);
        this.formControl.markAsTouched();
        if (this.closeAfterValueSelected) {
          this._closeOverlay(overlayRef);
          this._tabulationService.focusNext(this._elementToReFocus);
        } else {
          overlayRef.updatePosition();
        }
      }),
    ));
  }

  protected abstract extraBehaviorOnValueSelectedInOvertly(value: TResource): void;

  private _observeCloseEventInOverlay(componentRef: ComponentRef<SharedAbstractContentPresentational<TResource>>, overlayRef: OverlayRef): void {
    this.subscribe(componentRef.instance.closeObs.pipe(
      tap(() => {
        this._closeOverlay(overlayRef);
        this._tabulationService.focusNext(this._elementToReFocus);
      }),
    ));
  }

  private _observeClickBackdrop(componentRef: ComponentRef<SharedAbstractContentPresentational<TResource>>, overlayRef: OverlayRef): void {

    this.subscribe(overlayRef.backdropClick().pipe(
      tap(() => {
        this._closeOverlay(overlayRef);
        this._tabulationService.focusNext(this._elementToReFocus);
      }),
    ));
    this.subscribe(componentRef.instance.closeByTabObs.pipe(
      tap(() => {
          this._closeOverlay(overlayRef);
          this._tabulationService.focusNext(this._elementToReFocus);
        },
      )));
    this.subscribe(componentRef.instance.closeByShiftTabObs.pipe(
      tap(() => {
          this._closeOverlay(overlayRef);
          this._tabulationService.focusPrev(this._elementToReFocus);
        },
      )));
  }

  private _closeOverlay(overlayRef: OverlayRef, keepInOverlayRefs?: boolean): void {
    overlayRef.detach();
    if (!keepInOverlayRefs) {
      this._overlayRefs.delete(overlayRef);
    }
    this.formControl.markAsTouched();
    this._changeDetector.detectChanges();
  }

  private _getPosition(elementToConnectTo: ElementRef): FlexibleConnectedPositionStrategy {
    return this._overlay.position()
      .flexibleConnectedTo(elementToConnectTo)
      .withFlexibleDimensions(false)
      .withPositions([
        {
          originX: "start",
          originY: "bottom",
          overlayX: "start",
          overlayY: "top",
        },
      ]);
  }

  focusInputIfNotClickTarget($event: MouseEvent): void {
    if (!isNullOrUndefined(this.input) && $event.target !== this.input.nativeElement) {
      this.input.nativeElement.focus();
    }
  }

  add(item: TResource): void {
    if (!isNullOrUndefined(this.searchItemInSelectedItems(item))) {
      return;
    }
    this.selectedItems.push(item);
    this._changeDetector.detectChanges();
    this._updateFormControlWithSelectedItem();
  }

  remove(item: TResource): void {
    const index = this.searchIndexItemInSelectedItems(item);
    if (index === -1) {
      return;
    }
    this.selectedItems.splice(index, 1);
    this._updateFormControlWithSelectedItem();
    this._changeDetector.detectChanges();
  }

  private _updateFormControlWithSelectedItem(): void {
    const arrayKeys = this.selectedItems.map(c => c[this.valueKey]);
    this.formControl.setValue(arrayKeys);
    this.propagateChange(arrayKeys);
    this._valueBS.next(arrayKeys);
  }

  protected searchItemInSelectedItemsWithId(id: string): TResource | undefined {
    const index = this.searchIndexItemInSelectedItems({[this.valueKey]: id} as any);
    if (index === -1) {
      return this.selectedItems[index];
    }
  }

  private searchItemInSelectedItems(item: TResource): TResource | undefined {
    const index = this.searchIndexItemInSelectedItems(item);
    if (index === -1) {
      return undefined;
    }
    return this.selectedItems[index];
  }

  private searchIndexItemInSelectedItems(item: TResource): number {
    return this.selectedItems.findIndex(s => s[this.valueKey] === item[this.valueKey]);
  }

  getTooltip(value: TResource): string {
    let extraInfo = StringUtil.stringEmpty;
    if (!isNullOrUndefined(this.extraInfoLabelKey) && !isNullOrUndefined(value[this.extraInfoLabelKey])) {
      extraInfo = " - " + value[this.extraInfoLabelKey];
    }
    return this.labelCallback(value) + extraInfo;
  }

  navigateTo(event: Event, value: TResource): void {
    if (isNotNullNorUndefined(event)) {
      event.stopPropagation();
      event.preventDefault();
    }
    this._navigateBS.next(value);
  }

  copyToClipboard(value: string): void {
    ClipboardUtil.copyStringToClipboard(value);
    this._notificationService.showInformation(MARK_AS_TRANSLATABLE("app.notification.extraInfoCopyToClipboard"));
  }

  closeOverlays(): void {
    this._overlayRefs.forEach(overlayRef => this._closeOverlay(overlayRef, true));
    this._overlayRefs.clear();
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() search: string;
  @PropertyName() value: string;
}
