import {
  ChangeDetectorRef,
  Directive,
  ElementRef,
  HostBinding,
  HostListener,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  AbstractControl,
  FormControl,
  FormGroup,
  ValidationErrors,
  Validators,
} from "@angular/forms";
import {LocalModelAttributeEnum} from "@app/shared/enums/model-attribute.enum";
import {environment} from "@environments/environment";
import {Navigate} from "@ngxs/router-plugin";
import {CanDeactivatePresentational} from "@shared/components/presentationals/shared-can-deactivate/shared-can-deactivate.presentational";
import {DomHelper} from "@shared/helpers/dom.helper";
import {FormControlKey} from "@shared/models/form-control-key.model";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  BaseResourceType,
  FormValidationHelper,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  ModelFormControlEvent,
  ObservableUtil,
  StringUtil,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractFormPresentational<TResource extends BaseResourceType> extends CanDeactivatePresentational implements OnInit {
  form: FormGroup;

  @HostBinding("class.edit-available")
  @Input()
  editAvailable: boolean;

  @Input()
  model: TResource;

  @HostBinding("class.readonly")
  private _readonly: boolean;

  protected _listFieldNameToDisplayErrorInToast: string[] = [];

  @Input()
  set readonly(value: boolean) {
    this._readonly = value;
    this.updateFormReadonly();
  }

  get readonly(): boolean {
    return this._readonly;
  }

  classInputIgnored: string = environment.classInputIgnored;
  isValidWhenDisable: boolean;

  private readonly _timeoutMsBeforeFocus: number = 100;
  private readonly TAG_MAT_FORM_FIELD: string = "mat-form-field";
  private readonly TAG_MAT_CHECKBOX: string = "mat-checkbox";
  private readonly TAG_TABLE_PERSON_ORGUNIT_ROLE: string = "dlcm-shared-table-person-orgunit-role-container";
  private readonly TAG_TEXTAREA: string = "textarea";
  private readonly TAG_MAT_SELECT: string = "mat-select";
  private readonly TAG_INPUT: string = "input";

  @HostListener("keydown.F2", ["$event"])
  onEnterInEdition(keyboardEvent: KeyboardEvent): void {
    if (keyboardEvent && !keyboardEvent.defaultPrevented) {
      keyboardEvent.preventDefault();
      if (!this.readonly || !this.editAvailable) {
        return;
      }
      this.enterInEditMode();
    }
  }

  @HostListener("click", ["$event"]) click(mouseEvent: MouseEvent): void {
    if (!this.readonly || !this.editAvailable) {
      return;
    }
    setTimeout(() => {
      if (isNonEmptyString(this._getHighlightedText())) {
        return;
      }
      const parentElement = DomHelper.getParentWithTag(mouseEvent.target as Element, [this.TAG_MAT_FORM_FIELD, this.TAG_MAT_CHECKBOX, this.TAG_TABLE_PERSON_ORGUNIT_ROLE], this._elementRef.nativeElement);
      if (isNotNullNorUndefined(parentElement) && !parentElement.classList.contains(CLASS_IGNORE_EDIT_FIELD)) {
        this.enterInEditMode();
        setTimeout(() => {
          const elementToFocus = this.getElementToFocus(parentElement);
          SharedAbstractFormPresentational._focus(elementToFocus);
        }, this._timeoutMsBeforeFocus);
      }
    });
  }

  private _getHighlightedText(): string {
    if (isNotNullNorUndefined(typeof window.getSelection)) {
      return window.getSelection().toString();
    } else if (isNotNullNorUndefined(typeof document?.["selection"]) && document?.["selection"]?.type === "Text") {
      return document["selection"].createRange().text;
    }
    return StringUtil.stringEmpty;
  }

  private getElementToFocus(element: Element): Element {
    if (element.tagName.toLowerCase() === this.TAG_MAT_FORM_FIELD) {
      const input = DomHelper.getParentChildrenWithTag(element, [`${this.TAG_INPUT}:not(.${environment.classInputIgnored})`, this.TAG_TEXTAREA, this.TAG_MAT_SELECT]);
      if (!isNullOrUndefined(input)) {
        return input;
      }
    }
    if (element.tagName.toLowerCase() === this.TAG_MAT_CHECKBOX) {
      return element;
    }
    return element;
  }

  private static _focus(element: Element): void {
    element["focus"]();
  }

  protected readonly _submitBS: BehaviorSubject<ModelFormControlEvent<TResource> | undefined> = new BehaviorSubject<ModelFormControlEvent<TResource> | undefined>(undefined);
  @Output("submitChange")
  readonly submitObs: Observable<ModelFormControlEvent<TResource> | undefined> = ObservableUtil.asObservable(this._submitBS);

  private readonly _dirtyBS: BehaviorSubject<boolean | undefined> = new BehaviorSubject<boolean | undefined>(undefined);
  @Output("dirtyChange")
  readonly dirtyObs: Observable<boolean | undefined> = ObservableUtil.asObservable(this._dirtyBS);

  private readonly _checkAvailableBS: BehaviorSubject<FormControlKey> = new BehaviorSubject<FormControlKey>(undefined);
  @Output("checkAvailableChange")
  readonly checkAvailableObs: Observable<FormControlKey> = ObservableUtil.asObservable(this._checkAvailableBS);

  protected readonly _navigateBS: BehaviorSubject<string[]> = new BehaviorSubject<string[]>(undefined);
  @Output("navigate")
  readonly navigateObs: Observable<string[]> = ObservableUtil.asObservable(this._navigateBS);

  protected readonly _navigateWithQueryParamBS: BehaviorSubject<Navigate> = new BehaviorSubject<Navigate>(undefined);
  @Output("navigateWithQueryParam")
  readonly navigateWithQueryParamObs: Observable<Navigate> = ObservableUtil.asObservable(this._navigateWithQueryParamBS);

  protected readonly _editBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("editChange")
  readonly editObs: Observable<void> = ObservableUtil.asObservable(this._editBS);

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  required: ValidationErrors = Validators.required;

  protected constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
                        protected readonly _elementRef: ElementRef) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.resetFormToInitialValue();
    this.updateFormReadonly();

    this.subscribe(this.form.statusChanges, () => {
      if (this._dirtyBS.value !== this.form.dirty) {
        this._dirtyBS.next(this.form.dirty);
      }
    });
  }

  navigate(value: string[]): void {
    this._navigateBS.next(value);
  }

  resetFormToInitialValue(): void {
    if (this.model) {
      this.bindFormTo(this.model);
    } else {
      this.initNewForm();
    }
  }

  private updateFormReadonly(): void {
    if (isNullOrUndefined(this.form)) {
      return;
    }
    if (this.readonly) {
      this.form.disable();
    } else {
      this.form.enable();
      this.disableSpecificField();
    }
  }

  protected disableSpecificField(): void {
    // OVERRIDE THIS METHOD IF NECESSARY
  }

  protected abstract initNewForm(): void;

  protected abstract bindFormTo(model: TResource): void;

  onSubmit(): void {
    const model = this.treatmentBeforeSubmit(this.form.value as TResource);
    this.addResId(model);
    this._submitBS.next({
      model: model,
      formControl: this.form,
      changeDetectorRef: this._changeDetectorRef,
      listFieldNameToDisplayErrorInToast: this._listFieldNameToDisplayErrorInToast,
    });
  }

  protected abstract treatmentBeforeSubmit(model: TResource): TResource;

  protected addResId(model: TResource): void {
    const isUpdate = this.model !== null && this.model !== undefined;
    if (isUpdate) {
      model[LocalModelAttributeEnum.resId] = this.model[LocalModelAttributeEnum.resId];
    }
  }

  canDeactivate(): boolean {
    return !this.form.dirty;
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  isRequired(key: string): boolean {
    const errors = this.getFormControl(key).errors;
    return isNullOrUndefined(errors) ? false : errors.required;
  }

  checkAvailable(key: string, formControl: FormControl): void {
    this._checkAvailableBS.next({key: key, formControl: formControl, changeDetector: this._changeDetectorRef});
  }

  checkAvailableMultiple(...keys: string[]): void {
    this._checkAvailableBS.next({keys: keys, form: this.form, changeDetector: this._changeDetectorRef});
  }

  enterInEditMode(): void {
    this._editBS.next();
  }
}

export const CLASS_IGNORE_EDIT_FIELD: string = environment.classInputIgnored;
