import {
  ChangeDetectorRef,
  Directive,
  Inject,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {
  MAT_DIALOG_DATA,
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {SharedAbstractCrudRoutable} from "@app/shared/components/routables/shared-abstract-crud/shared-abstract-crud.routable";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {ActionCompletion} from "@ngxs/store/src/operators/of-action";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {CrudHelper} from "@shared/helpers/crud.helper";
import {CreateEditDialog} from "@shared/models/detail-edit-dialog.model";
import {FormControlKey} from "@shared/models/form-control-key.model";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  BaseResourceType,
  isNullOrUndefined,
  isTrue,
  MARK_AS_TRANSLATABLE,
  ModelFormControlEvent,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractCreateEditDialog<TResourceModel extends BaseResourceType, UResourceStateModel extends ResourceStateModel<TResourceModel>> extends SharedAbstractCrudRoutable<TResourceModel, UResourceStateModel> implements OnInit, OnDestroy {
  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  readonly KEY_UPDATE_BUTTON: string = MARK_AS_TRANSLATABLE("dialog.createEdit.button.update");
  readonly KEY_CANCEL_BUTTON: string | undefined = MARK_AS_TRANSLATABLE("dialog.createEdit.button.cancel");
  readonly KEY_CREATE_TITLE: string | undefined = MARK_AS_TRANSLATABLE("dialog.createEdit.title.create");
  readonly KEY_UPDATE_TITLE: string | undefined = MARK_AS_TRANSLATABLE("dialog.createEdit.title.update");

  current: TResourceModel | undefined;
  isLoadingObs: Observable<boolean> = this._store.select(s => StoreUtil.isLoadingState(super.getState(s)));

  get isUpdateMode(): boolean {
    return !this.isCreateMode;
  }

  get isCreateMode(): boolean {
    return isNullOrUndefined(this.data.current);
  }

  @ViewChild("formPresentational")
  readonly formPresentational: SharedAbstractFormPresentational<any>;

  protected constructor(@Inject(MAT_DIALOG_DATA) protected readonly data: CreateEditDialog<TResourceModel>,
                        protected readonly _store: Store,
                        protected readonly _route: ActivatedRoute,
                        protected readonly _actions$: Actions,
                        protected readonly _changeDetector: ChangeDetectorRef,
                        protected readonly _dialog: MatDialog,
                        protected readonly _dialogRef: MatDialogRef<SharedAbstractCreateEditDialog<TResourceModel, UResourceStateModel>>,
                        protected readonly _state: LocalStateEnum,
                        protected readonly _resourceNameSpace: ResourceNameSpace,
                        protected readonly _parentState?: LocalStateEnum) {
    super(_store, _state, _parentState);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.current = this.data.current;
    this.retrieveResource();
    if (!isNullOrUndefined(this.current)) {
      this.getSubResourceWithParentId(this.current.resId);
    }
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }

  protected retrieveResource(): void {
    this._store.dispatch(ResourceActionHelper.loadResource(this._resourceNameSpace));
  }

  protected cleanState(): void {
    this._store.dispatch(ResourceActionHelper.clean(this._resourceNameSpace, false, new QueryParameters()));
  }

  protected abstract getSubResourceWithParentId(id: string): void;

  update(model: ModelFormControlEvent<TResourceModel>): Observable<any> {
    super.saveInProgress();
    if (this.isUpdateMode) {
      return this._store.dispatch(ResourceActionHelper.update<TResourceModel>(this._resourceNameSpace, model));
    }
    if (this.isCreateMode) {
      return this._store.dispatch(ResourceActionHelper.create<TResourceModel>(this._resourceNameSpace, model));
    }
  }

  confirm(): void {
    this.formPresentational.onSubmit();

    if (this.isUpdateMode) {
      this.subscribe(this._actions$.pipe(
        ofActionCompleted(this._resourceNameSpace.UpdateSuccess),
        tap((result) => {
          this.closeDialogIfSuccess(result);
        }),
      ));
    }

    if (this.isCreateMode) {
      this.subscribe(this._actions$.pipe(
        ofActionCompleted(this._resourceNameSpace.CreateSuccess),
        tap((result) => {
          this.closeDialogIfSuccess(result);
        }),
      ));
    }
  }

  private closeDialogIfSuccess(result: ActionCompletion<boolean>): void {
    if (isTrue(result.result.successful)) {
      this._dialogRef.close(true);
    }
  }

  checkAvailable(formControlKey: FormControlKey): void {
    this.subscribe(CrudHelper.checkAvailable(formControlKey, this._store, this._actions$, this._resourceNameSpace, this.current.resId));
  }
}
