import {
  animate,
  state,
  style,
  transition,
  trigger,
} from "@angular/animations";
import {DOCUMENT} from "@angular/common";
import {
  ChangeDetectionStrategy,
  Component,
  HostBinding,
  Inject,
  OnInit,
  Renderer2,
} from "@angular/core";
import {AppAction} from "@app/stores/app.action";
import {AppState} from "@app/stores/app.state";
import {AppTocState} from "@app/stores/toc/app-toc.state";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {TocHelper} from "@shared/helpers/toc.helper";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  MemoizedUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-userguide-sidebar-container",
  templateUrl: "./shared-userguide-sidebar.container.html",
  styleUrls: ["./shared-userguide-sidebar.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [
    trigger("sidebarAnimation", [
      state("void", style({opacity: "0" /*right: "-400px"*/})),
      transition(":enter", animate("300ms ease")),
      transition(":leave", animate("300ms ease")),
    ]),
    trigger("backdropAnimation", [
      state("void", style({opacity: "0"})),
      transition(":enter", animate("300ms ease")),
      transition(":leave", animate("300ms ease")),
    ]),
  ],
})
export class SharedUserguideSidebarContainer extends SharedAbstractContainer implements OnInit {
  userDocumentationObs: Observable<string> = MemoizedUtil.select(this._store, AppTocState, s => s.userDocumentation);
  isOpenedUserGuideSidebarObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, s => s.isOpenedSidebarUserGuide);

  @HostBinding("class.hide")
  hide: boolean = true;

  constructor(private readonly _store: Store,
              private readonly _renderer: Renderer2,
              protected readonly _actions$: Actions,
              @Inject(DOCUMENT) private readonly _document: Document) {
    super();
  }

  ngOnInit(): void {
    this.subscribe(this.userDocumentationObs.pipe(
      distinctUntilChanged(),
      filter(doc => isNotNullNorUndefined(doc)),
      take(1),
      ),
      () => this.hide = false);

    this.subscribe(this._actions$.pipe(
      ofActionCompleted(AppAction.ChangeDisplaySidebarUserGuide),
      filter(action => (action.action as AppAction.ChangeDisplaySidebarUserGuide).isOpen),
      tap(action => this._updateLinkUserGuide()),
    ));
  }

  private _updateLinkUserGuide(): void {
    setTimeout(() => {
      TocHelper.updateLinkToc(this._document, this._renderer, "user-guide-sidebar", true);
    }, 0);
  }

  closeUserGuideSidebar(): void {
    this._store.dispatch(new AppAction.ChangeDisplaySidebarUserGuide(false));
  }
}
