import {
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  ActivatedRoute,
  NavigationCancel,
  NavigationEnd,
  Router,
} from "@angular/router";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {DataTestEnum} from "@shared/enums/data-test.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {ObservableOrPromiseOrValue} from "@shared/models/extra-button-toolbar.model";
import {
  BehaviorSubject,
  Observable,
  of,
} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  tap,
} from "rxjs/operators";
import {
  isArray,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isObservable,
  ObservableUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-tabs-container",
  templateUrl: "./shared-tabs.container.html",
  styleUrls: ["./shared-tabs.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedTabsContainer extends SharedAbstractContainer implements OnInit {
  @Input()
  tabs: Tab[];

  @Input()
  initialTabActiveId: string | undefined;

  @Input()
  isLoading: boolean;

  @Input()
  suffixUrlMatcher: (route: ActivatedRoute) => string = (route: ActivatedRoute) => {
    const children = route.snapshot.children;
    if (children.length > 0 && children[0].url.length > 0) {
      const mode = children[0].url[0].path;
      return mode;
    }
    return undefined;
    // tslint:disable-next-line:semicolon
  };

  private readonly _tabBS: BehaviorSubject<Tab> = new BehaviorSubject<Tab>(undefined);
  @Output("tabChange")
  readonly tabObs: Observable<Tab> = ObservableUtil.asObservable(this._tabBS);

  currentTab: Tab;

  @Input()
  tabTourAnchor: TourEnum;

  constructor(private readonly _store: Store,
              private readonly _router: Router,
              private readonly _route: ActivatedRoute) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    if (isNotNullNorUndefined(this.initialTabActiveId)) {
      this.currentTab = this.tabs.find(t => t.id === this.initialTabActiveId);
      this._tabBS.next(this.currentTab);
    }
    this._computeCurrentTabFromRoute();
    this.subscribe(this._router.events
      .pipe(
        filter(event => event instanceof NavigationCancel || event instanceof NavigationEnd),
        distinctUntilChanged(),
        tap(event => {
          this._computeCurrentTabFromRoute();
        }),
      ),
    );
  }

  trackByFn(index: number, tab: Tab): string {
    return tab.id;
  }

  private _computeCurrentTabFromRoute(): Tab | undefined {
    const tabRouteSelected = this.suffixUrlMatcher(this._route);
    const tabIndex = this.tabs.findIndex(t => t.suffixUrl === tabRouteSelected);
    if (tabIndex === -1) {
      return undefined;
    }
    this.currentTab = this.tabs[tabIndex];
    this._tabBS.next(this.currentTab);
    return this.currentTab;
  }

  private dispatchActionNavigateToTab(selectedTab: Tab): Observable<any> {
    const path = selectedTab.route();
    return this._store.dispatch(new Navigate(isArray(path) ? path : [path]));
  }

  isDisplayed(tab: Tab): Observable<boolean> {
    if (isNullOrUndefined(tab.conditionDisplay)) {
      return of(true);
    }
    if (isObservable(tab.conditionDisplay())) {
      return tab.conditionDisplay() as Observable<boolean>;
    } else {
      return of(tab.conditionDisplay() as boolean);
    }
  }

  navigateToTab(tab: Tab): void {
    if (this.currentTab === tab) {
      return;
    }
    if (isNullOrUndefined(tab.route)) {
      this.currentTab = tab;
      return;
    }
    this.dispatchActionNavigateToTab(tab);
  }
}

export interface Tab {
  id: string;
  suffixUrl?: string;
  icon?: IconNameEnum;
  titleToTranslate: string;
  conditionDisplay?: () => boolean | Observable<boolean> | undefined;
  route?: () => string | string[];
  numberNew?: () => ObservableOrPromiseOrValue<number>;
  dataTest?: DataTestEnum;
}
