import {PersonRole} from "@admin/models/person-role.model";
import {Relation3TiersPersonOrgunitRole} from "@admin/models/relation-3-tiers-person-orgunit-role.model";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  HostBinding,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  AbstractControl,
  ControlValueAccessor,
  FormArray,
  FormBuilder,
  FormControl,
  FormGroup,
  NG_VALUE_ACCESSOR,
  Validators,
} from "@angular/forms";
import {Enums} from "@enums";
import {
  OrganizationalUnit,
  Role,
} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {Store} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {JobExecutionReport} from "@shared/models/business/job-execution-report.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {sharedOrganizationalUnitActionNameSpace} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrganizationalUnitState} from "@shared/stores/organizational-unit/shared-organizational-unit.state";
import {sharedPersonActionNameSpace} from "@shared/stores/person/shared-person.action";
import {SharedPersonState} from "@shared/stores/person/shared-person.state";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {tap} from "rxjs/operators";
import {
  BaseResource,
  FormValidationHelper,
  isEmptyArray,
  isNullOrUndefined,
  isUndefined,
  MARK_AS_TRANSLATABLE,
  ObservableUtil,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  Sort,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-table-person-orgunit-role-container",
  templateUrl: "./shared-table-person-orgunit-role.container.html",
  styleUrls: ["./shared-table-person-orgunit-role.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedTablePersonOrgunitRoleContainer,
    },
  ],
})
export class SharedTablePersonOrgunitRoleContainer extends SharedAbstractContainer implements ControlValueAccessor, OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  formArray: FormArray;

  checkboxDisplayMode: "checkbox" | "radio" = "radio";

  columns: DataTableColumns<BaseResource & JobExecutionReport>[] = [
    {
      field: "person-or-orgunit",
      header: MARK_AS_TRANSLATABLE("person_or_org_unit"),
      type: FieldTypeEnum.string,
      order: OrderEnum.descending,
      isFilterable: false,
      isSortable: true,
    },
    {
      field: "VISITOR",
      header: MARK_AS_TRANSLATABLE("VISITOR"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "CREATOR",
      header: MARK_AS_TRANSLATABLE("CREATOR"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "APPROVER",
      header: MARK_AS_TRANSLATABLE("APPROVER"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "STEWARD",
      header: MARK_AS_TRANSLATABLE("STEWARD"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "MANAGER",
      header: MARK_AS_TRANSLATABLE("MANAGER"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
  ];

  @Input()
  mode: PersonOrgUnitRoleMode = PersonOrgUnitRoleMode.undefined;

  @HostBinding("class.disabled") disabled: boolean = false;

  @Input()
  selectedPersonOrOrgUnitRole: Relation3TiersPersonOrgunitRole[];

  @Input()
  listOrgUnit: OrganizationalUnit[];

  @Input()
  listRole: Role[];

  @Input()
  isWithLinkToAdmin: boolean = true;

  @Input()
  highlightedPersonOrOrgUnitId: string;

  sharedPersonSort: Sort = {
    field: "lastName",
    order: OrderEnum.ascending,
  };
  sharedPersonActionNameSpace: ResourceNameSpace = sharedPersonActionNameSpace;
  sharedPersonState: typeof SharedPersonState = SharedPersonState;

  sharedOrgUnitSort: Sort = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedOrgUnitActionNameSpace: ResourceNameSpace = sharedOrganizationalUnitActionNameSpace;
  sharedOrgUnitState: typeof SharedOrganizationalUnitState = SharedOrganizationalUnitState;

  @Input()
  formControl: FormControl;

  _readonly: boolean;

  @Input()
  set readonly(value: boolean) {
    this._readonly = value;
    this.updateFormReadonlyState();
  }

  get readonly(): boolean {
    return this._readonly;
  }

  private readonly _valueBS: BehaviorSubject<any[] | undefined> = new BehaviorSubject<any[] | undefined>(undefined);
  @Output("valueChange")
  readonly valueObs: Observable<any[] | undefined> = ObservableUtil.asObservable(this._valueBS);

  protected readonly _navigateBS: BehaviorSubject<string[]> = new BehaviorSubject<string[]>(undefined);
  @Output("navigate")
  readonly navigateObs: Observable<string[]> = ObservableUtil.asObservable(this._navigateBS);

  get modeEnum(): typeof PersonOrgUnitRoleMode {
    return PersonOrgUnitRoleMode;
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  getFormControl(formControl: AbstractControl, key: string): AbstractControl {
    return formControl.get(key);
  }

  isRequired(formControl: AbstractControl, key: string): boolean {
    const errors = formControl.get(key).errors;
    return isNullOrUndefined(errors) ? false : errors.required;
  }

  constructor(private translate: TranslateService,
              private store: Store,
              private readonly _fb: FormBuilder,
              private readonly _changeDetector: ChangeDetectorRef,
              private readonly _elementRef: ElementRef) {
    super();
  }

  private markAsTouched(value: Model[]): void {
    this.propagateChange(value);
    this._valueBS.next(value);
    this.formControl.markAllAsTouched();
  }

  ngOnInit(): void {
    super.ngOnInit();

    // TODO Manage better form to use directly formControl (allow to share disable and valid state)
    this.formArray = this._fb.array([]);

    if (!isNullOrUndefined(this.selectedPersonOrOrgUnitRole)) {
      this.selectedPersonOrOrgUnitRole.forEach(p => {
        this.formArray.push(this.createPersonOrOrgunitRole(p));
      });
    }

    this.updateFormReadonlyState();
    this.formControl.setValue(this.formArray.value);

    const valueBefore: Model[] = this.formArray.value;
    this.subscribe(this.formArray.valueChanges.pipe(
      tap((value: Model[]) => {
        if (valueBefore.length !== value.length) {
          this.markAsTouched(value);
          return;
        }

        let isDiff = false;
        valueBefore.forEach((valBefore, index) => {
          if (isDiff) {
            return;
          }
          if (valBefore.id !== value[index].id || valBefore.listId !== value[index].listId) {
            this.markAsTouched(value);
            isDiff = true;
            return;
          }
        });

        if (!isDiff) {
          this.formControl.markAsPristine();
        }
      }),
    ));

    this.subscribe(this.formArray.statusChanges.pipe(
      tap(status => {
        if (status === "VALID") {
          this.formControl.setErrors(null);
        } else if (status === "INVALID") {
          this.formControl.setErrors({"invalid": true});
        }
      }),
    ));
  }

  private updateFormReadonlyState(): void {
    if (isNullOrUndefined(this.formArray) || isNullOrUndefined(this.formControl)) {
      return;
    }
    if (this._readonly) {
      this.disabled = true;
      this.formArray.disable();
      this.formControl.disable();
    } else {
      this.disabled = false;
      this.formArray.enable();
      this.formControl.enable();
    }
  }

  createPersonOrOrgunitRole(person: PersonRole | undefined = undefined): FormGroup {
    if (isUndefined(person)) {
      const form = this._fb.group({
        [this.formDefinition.id]: ["", [Validators.required]],
        [this.formDefinition.listId]: ["", [Validators.required]],
      });
      return form;
    } else {
      // We transform here list of roles into unique role because we hide the multiples roles to the final user
      const listId = person.roles.length > 0 ? this.getMostImportantRole(person.roles.map(r => r.resId) as Enums.Role.RoleEnum[]) : "";
      const form = this._fb.group({
        [this.formDefinition.id]: [person.resId, [Validators.required]],
        [this.formDefinition.listId]: [listId, [Validators.required]],
      });
      if (person.isReadonly) {
        setTimeout(() => {
          form.disable();
        }, 0);
      }
      return form;
    }
  }

  addPersonOrOrgUnit(tryLater: boolean = true): void {
    if (!this.formControl.disabled) {
      this.formArray.push(this.createPersonOrOrgunitRole());
      // TODO Focus newly line created
      // setTimeout(() => {
      //   const inputToFocus = this._elementRef.nativeElement.querySelector("table tbody tr:last-child > td:first-child > input");
      //   inputToFocus.focus();
      // }, 0);
    } else {
      if (tryLater) {
        setTimeout(() => this.addPersonOrOrgUnit(false), 10);
      }
    }
  }

  delete(index: number): void {
    this.formArray.removeAt(index);
  }

  private filter(listPersonOrOrgUnit: OrganizationalUnit[], formControl: AbstractControl): any {
    if (isNullOrUndefined(listPersonOrOrgUnit) || isEmptyArray(listPersonOrOrgUnit)) {
      return listPersonOrOrgUnit;
    }
    const personOrOrgUnitAlreadySelected = this.formArray.controls.map(c => c.get(this.formDefinition.id).value);
    return listPersonOrOrgUnit.filter(p => p.resId === formControl.value || !personOrOrgUnitAlreadySelected.includes(p.resId));
  }

  propagateChange = (__: any) => {};

  registerOnChange(fn: any): void {
    this.propagateChange = fn;
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState(isDisabled: boolean): void {
  }

  writeValue(value: string[]): void {
    // if (value) {
    // this.formControl.setValue(value);
    // }
  }

  getMessageNoItem(): string {
    if (this.mode === PersonOrgUnitRoleMode.person) {
      return MARK_AS_TRANSLATABLE("shared.personOrOrgUnitRole.noItem.person");
    }
    if (this.mode === PersonOrgUnitRoleMode.orgUnit) {
      return MARK_AS_TRANSLATABLE("shared.personOrOrgUnitRole.noItem.orgUnit");
    }
  }

  private getMostImportantRole(listRoles: Enums.Role.RoleEnum[]): Enums.Role.RoleEnum | string {
    if (listRoles.length === 0) {
      return "";
    }
    if (listRoles.length === 1) {
      return listRoles[0];
    }
    let role = listRoles.find(r => r === Enums.Role.RoleEnum.MANAGER);
    if (!isNullOrUndefined(role)) {
      return Enums.Role.RoleEnum.MANAGER;
    }
    role = listRoles.find(r => r === Enums.Role.RoleEnum.STEWARD);
    if (!isNullOrUndefined(role)) {
      return Enums.Role.RoleEnum.STEWARD;
    }
    role = listRoles.find(r => r === Enums.Role.RoleEnum.APPROVER);
    if (!isNullOrUndefined(role)) {
      return Enums.Role.RoleEnum.APPROVER;
    }
    role = listRoles.find(r => r === Enums.Role.RoleEnum.CREATOR);
    if (!isNullOrUndefined(role)) {
      return Enums.Role.RoleEnum.CREATOR;
    }
    role = listRoles.find(r => r === Enums.Role.RoleEnum.VISITOR);
    if (!isNullOrUndefined(role)) {
      return Enums.Role.RoleEnum.VISITOR;
    }
    return StringUtil.stringEmpty;
  }

  goToPerson(personId: string): void {
    this._navigateBS.next([RoutesEnum.adminPersonDetail, personId]);
  }

  goToOrgUnit(orgUnitId: string): void {
    this._navigateBS.next([RoutesEnum.adminOrganizationalUnitDetail, orgUnitId]);
  }

  changeRole(f: AbstractControl, role: Role, $event: MouseEvent): void {
    const formControl = this.getFormControl(f, this.formDefinition.listId);
    if (formControl.value !== role.resId) {
      this.getFormControl(f, this.formDefinition.listId).setValue(role.resId);
    }
    $event.preventDefault();
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() id: string;
  @PropertyName() listId: string;
}

interface Model {
  id: string;
  listId: string;
}

export enum PersonOrgUnitRoleMode {
  undefined,
  person,
  orgUnit,
}
