import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
} from "@angular/core";
import {environment} from "@environments/environment";
import {OrganizationalUnit} from "@models";
import {Store} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedOrgUnitAction} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrganizationalUnitState} from "@shared/stores/organizational-unit/shared-organizational-unit.state";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  MemoizedUtil,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-organizational-unit-name-container",
  templateUrl: "./shared-organizational-unit-name.container.html",
  styleUrls: ["./shared-organizational-unit-name.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedOrganizationalUnitNameContainer extends SharedAbstractContainer {
  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnInput;

  name: string;
  listOrgUnitObs: Observable<OrganizationalUnit[]> = MemoizedUtil.list(this._store, SharedOrganizationalUnitState);
  isLoading: boolean = false;

  _id: string;

  @Input()
  set id(id: string) {
    this._id = id;
    if (isNullOrUndefined(id)) {
      this.name = StringUtil.stringEmpty;
      return;
    }
    const listSnapshot = MemoizedUtil.listSnapshot(this._store, SharedOrganizationalUnitState);
    const orgUnit = this.getOrgUnit(listSnapshot);
    if (isNullOrUndefined(orgUnit)) {
      this.name = StringUtil.stringEmpty;
      this.subscribe(this.listOrgUnitObs.pipe(
        distinctUntilChanged(),
        filter(list => isNotNullNorUndefined(this.getOrgUnit(list))),
        take(1),
      ), list => {
        this.name = this.getOrgUnit(list)?.name;
        this.isLoading = false;
        this._changeDetector.detectChanges();
      });
      this._store.dispatch(new SharedOrgUnitAction.AddInListById(id, true));
      this.isLoading = true;
    } else {
      this.name = orgUnit.name;
    }
  }

  get id(): string {
    return this._id;
  }

  @Input()
  inline: boolean = false;

  constructor(public readonly _store: Store,
              private readonly _changeDetector: ChangeDetectorRef) {
    super();
  }

  private getOrgUnit(list: OrganizationalUnit[]): OrganizationalUnit {
    if (isNullOrUndefined(list)) {
      return undefined;
    }
    const organizationalUnit = list.find(o => o.resId === this._id);
    if (isNullOrUndefined(organizationalUnit)) {
      return undefined;
    }
    return organizationalUnit;
  }
}
