import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnChanges,
  SimpleChanges,
} from "@angular/core";
import {
  DomSanitizer,
  SafeResourceUrl,
} from "@angular/platform-browser";
import {User} from "@models";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {Observable} from "rxjs/index";
import {
  distinctUntilChanged,
  filter,
  takeWhile,
  tap,
} from "rxjs/operators";
import {
  BaseResource,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObjectUtil,
  MemoizedUtil,
} from "solidify-frontend";
import {MappingObject} from "solidify-frontend/lib/models";

@Component({
  selector: "dlcm-shared-logo-wrapper-container",
  templateUrl: "./shared-logo-wrapper.container.html",
  styleUrls: ["./shared-logo-wrapper.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedLogoWrapperContainer<TResource extends BaseResource> extends SharedAbstractContainer implements OnChanges {
  _idResource: string;
  isLoading: boolean = true;

  @Input()
  logoNamespace: ResourceLogoNameSpace;

  @Input()
  logoState: any;

  @Input()
  isUser: boolean = false;

  @Input()
  user: User;

  @Input()
  set id(id: string) {
    this._idResource = id;
  }

  get id(): string {
    return this._idResource;
  }

  listPhotosObs: Observable<MappingObject<string>>;

  photo: string;

  constructor(protected readonly _store: Store,
              protected readonly _actions$: Actions,
              private readonly _changeDetector: ChangeDetectorRef,
              protected readonly _sanitizer: DomSanitizer) {
    super();
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (isNotNullNorUndefined(this.logoNamespace)
      && isNotNullNorUndefined(changes.id.currentValue)
      && isNotNullNorUndefined(changes.user.currentValue)) {

      this.photo = undefined;
      this.isLoading = true;

      //in case of user, the id_resource has to be changed from user to person.resId
      if (this.isUser && !isNullOrUndefined(this.user.person)) {
        this._idResource = this.user.person.resId;
      }
      this.listPhotosObs = MemoizedUtil.select(this._store, this.logoState, (state: ResourceLogoStateModel<TResource>) => state.listLogo);
      this.getLogo();
    }
  }

  protected getLogo(): void {
    this._store.dispatch(new this.logoNamespace.GetPhotoByResId(this._idResource));

    const listPhotos = this._store.selectSnapshot((state: ResourceLogoStateModel<TResource>) => state.listLogo);
    const photo = this.getPhoto(listPhotos);
    if (isNullOrUndefined(photo)) {
      this.subscribe(this.listPhotosObs.pipe(
        distinctUntilChanged(),
        filter(list => isNotNullNorUndefined(this.getPhoto(list))),
        takeWhile((list) => this.isLoading === true),
        tap((list) => {
          this.photo = MappingObjectUtil.get(MemoizedUtil.selectSnapshot(this._store, this.logoState, (state: ResourceLogoStateModel<TResource>) => state.listLogo), this._idResource);
          if (isNotNullNorUndefined(this.photo)) {
            this.isLoading = false;
            this._changeDetector.detectChanges();
          }
        })));
    }
  }

  protected sanitizeUrl(url: string | undefined): undefined | SafeResourceUrl {
    if (isNullOrUndefined(url)) {
      return undefined;
    } else {
      return this._sanitizer.bypassSecurityTrustResourceUrl(url);
    }
  }

  protected getPhoto(list: MappingObject<string>): string | undefined {
    if (isNullOrUndefined(list)) {
      return undefined;
    }

    const photo: string = MappingObjectUtil.get(list, this._idResource);
    if (isNullOrUndefined(photo)) {
      return undefined;
    }
    return photo;
  }

}
