import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Inject,
  Input,
  OnInit,
  Output,
  Type,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {FileVisualizerHelper} from "@shared/filevisualizer/helpers/file-visualizer.helper";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {
  AbstractFileVisualizer,
  FILE_VISUALIZERS,
} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {DataFile} from "@shared/models/business/data-file.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {StatusHistoryNamespace} from "@shared/stores/status-history/status-history-namespace.model";
import {
  StatusHistoryState,
  StatusHistoryStateModel,
} from "@shared/stores/status-history/status-history.state";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  BaseResource,
  ClipboardUtil,
  DateUtil,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  ObservableUtil,
  OrderEnum,
  QueryParameters,
  urlSeparator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-file-and-aip-information-container",
  templateUrl: "./shared-file-and-aip-information.container.html",
  styleUrls: ["./shared-file-and-aip-information.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedFileAndAipInformationContainer<TResource extends BaseResource> extends SharedAbstractContainer implements OnInit {
  columns: DataTableColumns[];
  previewAvailable: boolean;
  visualizationErrorMessage: string;

  private readonly LOGO_SIP: string = "SIP";
  private readonly LOGO_AIP: string = "AIP";
  private readonly LOGO_DIP: string = "DIP";
  private readonly FILENAME_SIP_PACKAGE: string = "sip-package";
  private readonly FILENAME_AIP_PACKAGE: string = "aip-package";
  private readonly FILENAME_DOWNLOAD: string = "download";

  private readonly _fullScreenBS: BehaviorSubject<boolean | undefined> = new BehaviorSubject<boolean | undefined>(undefined);
  readonly fullScreenObs: Observable<boolean | undefined> = ObservableUtil.asObservable(this._fullScreenBS);

  _data: SharedFileAndAipInformationDialogData<TResource>;

  @Input()
  set data(value: SharedFileAndAipInformationDialogData<TResource>) {
    this._data = value;
    this._computePreviewAvailable();
  }

  get data(): SharedFileAndAipInformationDialogData<TResource> {
    return this._data;
  }

  @Input()
  canEdit: boolean = false;

  get modeEnum(): typeof SharedFileAndAipDetailDialogModeEnum {
    return SharedFileAndAipDetailDialogModeEnum;
  }

  get dateUtil(): typeof DateUtil {
    return DateUtil;
  }

  get dataCategoryEnum(): typeof Enums.DataFile.DataCategoryEnum {
    return Enums.DataFile.DataCategoryEnum;
  }

  get dataTypeEnum(): typeof Enums.DataFile.DataTypeEnum {
    return Enums.DataFile.DataTypeEnum;
  }

  get statusEnum(): typeof Enums.DataFile.StatusEnum {
    return Enums.DataFile.StatusEnum;
  }

  private readonly _locationBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("locationChange")
  readonly locationObs: Observable<void> = ObservableUtil.asObservable(this._locationBS);

  private readonly _dataCategoryTypeBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("dataCategoryTypeChange")
  readonly dataCategoryTypeObs: Observable<void> = ObservableUtil.asObservable(this._dataCategoryTypeBS);

  private readonly _deleteBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("deleteChange")
  readonly deleteObs: Observable<void> = ObservableUtil.asObservable(this._deleteBS);

  isInFullscreenMode: boolean = false;
  zoomEnable: boolean = false;

  smallButtons: ExtraButtonToolbar<TResource>[] = [
    {
      displayCondition: () => this.previewAvailable && this.isInFullscreenMode,
      icon: IconNameEnum.fullScreenLeave,
      callback: () => this.leaveFullScreen(),
      labelToTranslate: () => LabelTranslateEnum.exitFullscreen,
      color: "primary",
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _dialog: MatDialog,
              protected readonly _actions$: Actions,
              protected readonly _breakpointService: BreakpointService,
              protected readonly _notificationService: NotificationService,
              protected readonly _changeDetector: ChangeDetectorRef,
              @Inject(FILE_VISUALIZERS) protected readonly _fileVisualizers: AbstractFileVisualizer[]) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.columns = [
      {
        field: "changeTime",
        header: LabelTranslateEnum.changeTime,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        filterEnum: Enums.History.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
        translate: true,
        width: "110px",
      },
      {
        field: "description",
        header: LabelTranslateEnum.description,
        type: FieldTypeEnum.number,
        order: OrderEnum.none,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "creatorName",
        header: LabelTranslateEnum.createdBy,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        width: "140px",
      },
    ];
    this._computePreviewAvailable();
  }

  private _computePreviewAvailable(): void {
    if (isNullOrUndefined(this.data)) {
      return;
    }
    if (this.previewAvailable) {
      return;
    }
    if (isNullOrUndefined(this.data.dataFile?.fileFormat)) {
      this.previewAvailable = false;
      return;
    }
    this.previewAvailable = this.data.mode === SharedFileAndAipDetailDialogModeEnum.file && FileVisualizerHelper.canHandle(this._fileVisualizers, {
      dataFile: this.data.dataFile,
      fileExtension: FileVisualizerHelper.getFileExtension(this.data.dataFile.fileName),
    });
    if (!this.previewAvailable && this.data.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      this.visualizationErrorMessage = FileVisualizerHelper.errorMessage(this._fileVisualizers, {
        dataFile: this.data.dataFile,
        fileExtension: FileVisualizerHelper.getFileExtension(this.data.dataFile.fileName),
      });
    }
  }

  enterFullScreen(): void {
    this.isInFullscreenMode = true;
    this.zoomEnable = true;
    this._changeDetector.detectChanges();
  }

  leaveFullScreen(): void {
    this.isInFullscreenMode = false;
    this.zoomEnable = false;
    this._changeDetector.detectChanges();
  }

  get breakpointService(): BreakpointService {
    return this._breakpointService;
  }

  getFileInput(): FileInput {
    return {
      dataFile: this.data.dataFile,
    };
  }

  getResId(): string | undefined {
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      return this.data.dataFile.resId;
    }
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.aip) {
      return this.data.aip.resId;
    }
    return undefined;
  }

  getData(): DataFile | Aip | undefined {
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      return this.data.dataFile;
    }
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.aip) {
      return this.data.aip;
    }
    return undefined;
  }

  getDataFileName(): string {
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      return this.data.dataFile.fileName;
    }
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.aip) {
      return this.data.aip.info.name;
    }
  }

  getComplianceLevel(): string {
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      return this.data.dataFile.complianceLevel;
    }
    if (this.data.mode === SharedFileAndAipDetailDialogModeEnum.aip) {
      return this.data.aip.info.complianceLevel;
    }
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: this.data.parentId,
        resourceResId: this.getResId(),
        name: this.getDataFileName(),
        statusHistory: MemoizedUtil.select(this._store, this.data.statusHistoryState, state => state.history),
        isLoading: MemoizedUtil.isLoading(this._store, this.data.statusHistoryState),
        queryParametersObs: MemoizedUtil.select(this._store, this.data.statusHistoryState, state => state.queryParameters),
        state: this.data.statusHistoryNamespace,
      } as StatusHistoryDialog,
    });
  }

  getPuidLink(puid: string): string {
    return environment.urlNationalArchivePronom + puid;
  }

  copyKey(): void {
    ClipboardUtil.copyStringToClipboard(this.getResId());
    this._notificationService.showInformation(MARK_AS_TRANSLATABLE("app.notification.idCopyToClipboard"));
  }

  pagination($event: QueryParameters): void {
    this._store.dispatch(new this.data.statusHistoryNamespace.ChangeQueryParameters($event, this.getResId(), this.data.parentId));
  }

  back(): void {
    const currentUrl = this._store.selectSnapshot((s: LocalStateModel) => s.router.state.url);
    const previousUrl = currentUrl.substring(0, currentUrl.lastIndexOf(urlSeparator));
    this._store.dispatch(new Navigate([previousUrl]));
  }

  editLocation(): void {
    this._locationBS.next();
  }

  editCategory(): void {
    this._dataCategoryTypeBS.next();
  }

  delete(): void {
    this._deleteBS.next();
  }

  changeFullScreen($event: boolean): void {
    this._fullScreenBS.next($event);
  }

  getLogoPackage(): string {
    const mode = this.data.mode;
    switch (mode) {
      case SharedFileAndAipDetailDialogModeEnum.aip:
        return this.LOGO_AIP;
      case SharedFileAndAipDetailDialogModeEnum.file:
        const fileName = (this.getData() as DataFile)?.fileName;
        if (fileName === this.FILENAME_SIP_PACKAGE) {
          return this.LOGO_SIP;
        }
        if (fileName === this.FILENAME_AIP_PACKAGE || fileName === this.FILENAME_DOWNLOAD) {
          return this.LOGO_AIP;
        }
        return this.LOGO_DIP;
      default:
        return "?";
    }
  }
}

export interface SharedFileAndAipInformationDialogData<TResource extends BaseResource> {
  parentId: string;
  mode: SharedFileAndAipDetailDialogModeEnum;
  dataFile?: DataFile;
  aip?: Aip;
  buttons: ExtraButtonToolbar<TResource>[];
  statusHistoryState?: Type<StatusHistoryState<StatusHistoryStateModel<TResource>, TResource>>;
  statusHistoryNamespace?: StatusHistoryNamespace;
}

export enum SharedFileAndAipDetailDialogModeEnum {
  file,
  aip,
}
