import {AdminUserState} from "@admin/user/stores/admin-user.state";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {MatDialog} from "@angular/material/dialog";
import {DomSanitizer} from "@angular/platform-browser";
import {ActivatedRoute} from "@angular/router";
import {AppUserState} from "@app/stores/user/app-user.state";
import {User} from "@models";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractImageUploadWrapperContainer} from "@shared/components/containers/shared-abstract-image-upload-wrapper/shared-abstract-image-upload-wrapper.container";
import {
  SharedUploadImageDialog,
  SharedUploadImageDialogData,
} from "@shared/components/dialogs/shared-upload-image/shared-upload-image.dialog";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  isEmptyString,
  isFalse,
  isInstanceOf,
  isNullOrUndefined,
  MemoizedUtil,
  ObservableUtil,
  Override,
  OverrideProperty,
  StoreUtil,
} from "solidify-frontend";
import {
  take,
  tap,
} from "rxjs/operators";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {
  AppUserAction,
} from "@app/stores/user/app-user.action";
import {adminUserActionNameSpace} from "@admin/user/stores/admin-user.action";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";

@Component({
  selector: "dlcm-shared-avatar-upload-wrapper-container",
  templateUrl: "./shared-avatar-upload-wrapper.container.html",
  styleUrls: ["./shared-avatar-upload-wrapper.container.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAvatarUploadWrapperContainer extends SharedAbstractImageUploadWrapperContainer implements OnInit {
  @Input()
  user: User;

  @OverrideProperty()
  @Input()
  resourceLogoState: typeof AppUserState | typeof AdminUserState;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _fb: FormBuilder,
              protected readonly _sanitizer: DomSanitizer,
              protected readonly _cd: ChangeDetectorRef,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog) {
    super(_store,
      _route,
      _fb,
      _sanitizer,
      _cd,
      _actions$,
      _dialog);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getUserPhoto();
  }


  protected _defineSharedImageDialogData(): SharedUploadImageDialogData {
    return {
      titleToTranslate: LabelTranslateEnum.uploadAvatar,
      imageChangedEvent: undefined,
      isLoadingObs: this.isLoadingObs,
      aspectRatio: 1,
      roundCropped: true,
      resizeToHeight: 145 * 2,
      resizeToWidth: 145 * 2,
    };
  }

  private getUserPhoto(): void {
    this.subscribe(
      StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(this._store, [
        {
          action: new this.resourceLogoNameSpace.GetPhoto(this.resId),
          subActionCompletions: [
            this._actions$.pipe(ofActionCompleted(this.resourceLogoNameSpace.GetPhotoSuccess)),
            this._actions$.pipe(ofActionCompleted(this.resourceLogoNameSpace.GetPhotoFail)),
          ],
        },
      ]).pipe(
        take(1),
        tap( () => {
            this.photoObs = MemoizedUtil.select(this._store, this.resourceLogoState, (state: ResourceLogoStateModel<any>) => state.logo);
        }),
      ),
    );
  }

  @Override()
  fileChangeEvent(event: any): void {
    if (this.resourceLogoNameSpace === adminUserActionNameSpace && this.user.externalUid === MemoizedUtil.selectSnapshot(this._store, AppUserState, state => state.current.externalUid)) {
      // need to refresh user profile
      this.dispatchFileEventAndRefreshProfile(event);
    } else {
      super.fileChangeEvent(event);
    }
  }

  private dispatchFileEventAndRefreshProfile(event: any): void {
    if (isNullOrUndefined(event.target.value) || isEmptyString(event.target.value)) {
      return;
    }
    this._sharedUploadImageDialogData.imageChangedEvent = event;
    this.subscribe(this._dialog.open(SharedUploadImageDialog, {
      width: "90%",
      data: this._sharedUploadImageDialogData as SharedUploadImageDialogData,
    }).afterClosed().pipe(
      tap((blob: Blob | undefined) => {
        if (isNullOrUndefined(blob)) {
          return;
        }
        this.subscribe(
          this._actions$.pipe(
            ofActionCompleted(this.resourceLogoNameSpace.UploadPhotoSuccess),
            take(1),
            tap(action => {
              if (isFalse(action.result.successful)) {
                return;
              }
              this._store.dispatch(new this.resourceLogoNameSpace.GetPhoto(this.resId));
              this._store.dispatch(new AppUserAction.GetCurrentUser());
              this._avatarChangeBS.next(true);
            }),
          ),
        );
        this._store.dispatch(new this.resourceLogoNameSpace.UploadPhoto(this.resId, blob as any));
        this.fileInput.nativeElement.value = null;
      }),
    ));
  }
}
