import {
  ChangeDetectorRef,
  Directive,
  ElementRef,
  Input,
  OnInit,
  ViewChild,
  Output,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {MatDialog} from "@angular/material/dialog";
import {
  DomSanitizer,
  SafeResourceUrl,
} from "@angular/platform-browser";
import {ActivatedRoute} from "@angular/router";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {
  SharedUploadImageDialog,
  SharedUploadImageDialogData,
} from "@shared/components/dialogs/shared-upload-image/shared-upload-image.dialog";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  isEmptyString,
  isFalse,
  isNullOrUndefined,
  MemoizedUtil,
  ObservableUtil
} from "solidify-frontend";
import {SharedAbstractContainer} from "../shared-abstract/shared-abstract.container";

@Directive()
export abstract class SharedAbstractImageUploadWrapperContainer extends SharedAbstractContainer implements OnInit {
  @Input()
  resourceLogoNameSpace: ResourceLogoNameSpace;

  @Input()
  abstract resourceLogoState: any; //  typeof AppUserState | typeof AdminUserState | typeof AdminOrganizationalUnitState; // TODO Find a better typing

  isLoadingObs: Observable<boolean>;
  photoObs: Observable<string | undefined>;

  @Input()
  resId: string;

  @ViewChild("fileInput", {static: false})
  fileInput: ElementRef;

  protected readonly _avatarChangeBS: BehaviorSubject<boolean> = new BehaviorSubject<boolean>(false);
  @Output("avatarChange")
  readonly avatarChangeObs: Observable<boolean | undefined> = ObservableUtil.asObservable(this._avatarChangeBS);

  protected _sharedUploadImageDialogData: SharedUploadImageDialogData;

  readonly filesAccepted: string[] = [".jpeg", ".png", ".jpg"];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _fb: FormBuilder,
              protected readonly _sanitizer: DomSanitizer,
              protected readonly _cd: ChangeDetectorRef,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.isLoadingObs = MemoizedUtil.select(this._store, this.resourceLogoState, (state: ResourceLogoStateModel<any>) => state.isLoadingLogo);
    this.photoObs = MemoizedUtil.select(this._store, this.resourceLogoState, (state: ResourceLogoStateModel<any>) => state.logo);
    this._sharedUploadImageDialogData = this._defineSharedImageDialogData();
  }

  protected abstract _defineSharedImageDialogData(): SharedUploadImageDialogData;

  delete(): void {
    this.subscribe(
      this._actions$.pipe(
        ofActionCompleted(this.resourceLogoNameSpace.DeletePhotoSuccess),
        take(1),
        tap(action => {
          if (isFalse(action.result.successful)) {
            return;
          }
        }),
      ),
    );
    this._store.dispatch(new this.resourceLogoNameSpace.DeletePhoto(this.resId));
  }

  openModal(): void {
    this.fileInput.nativeElement.click();
  }

  sanitizeUrl(url: string | undefined): undefined | SafeResourceUrl {
    if (isNullOrUndefined(url)) {
      return undefined;
    } else {
      return this._sanitizer.bypassSecurityTrustResourceUrl(url);
    }
  }

  fileChangeEvent(event: any): void {
    if (isNullOrUndefined(event.target.value) || isEmptyString(event.target.value)) {
      return;
    }
    this._sharedUploadImageDialogData.imageChangedEvent = event;
    this.subscribe(this._dialog.open(SharedUploadImageDialog, {
      width: "90%",
      data: this._sharedUploadImageDialogData as SharedUploadImageDialogData,
    }).afterClosed().pipe(
      tap((blob: Blob | undefined) => {
        if (isNullOrUndefined(blob)) {
          return;
        }
        this.subscribe(
          this._actions$.pipe(
            ofActionCompleted(this.resourceLogoNameSpace.UploadPhotoSuccess),
            take(1),
            tap(action => {
              if (isFalse(action.result.successful)) {
                return;
              }
              this._store.dispatch(new this.resourceLogoNameSpace.GetPhoto(this.resId));
              this._avatarChangeBS.next(true);
            }),
          ),
        );
        this._store.dispatch(new this.resourceLogoNameSpace.UploadPhoto(this.resId, blob as any));
        this.fileInput.nativeElement.value = null;
      }),
    ));
  }
}
