import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {
  PreservationSpaceOrganizationalUnitAction,
  preservationSpaceOrganizationalUnitActionNameSpace,
} from "@app/features/preservation-space/organizational-unit/stores/preservation-space-organizational-unit.action";
import {PreservationSpaceOrganizationalUnitStateModel} from "@app/features/preservation-space/organizational-unit/stores/preservation-space-organizational-unit.state";
import {AppUserState} from "@app/stores/user/app-user.state";
import {Enums} from "@enums";
import {OrganizationalUnit} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {
  OrgunitRequestAccessDialogResult,
  PreservationSpaceOrganizationalUnitRequestAccessDialog,
} from "@preservation-space/organizational-unit/components/dialogs/organizational-unit-request-access/preservation-space-organizational-unit-request-access.dialog";
import {
  OrgunitRequestCreationDialogResult,
  PreservationSpaceOrganizationalUnitRequestCreationDialog,
} from "@preservation-space/organizational-unit/components/dialogs/organizational-unit-request-creation/preservation-space-organizational-unit-request-creation.dialog";
import {SharedAbstractListRoutable} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {DataTestEnum} from "@shared/enums/data-test.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  PreservationSpaceOrganizationalUnitRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {SecurityService} from "@shared/services/security.service";
import {SharedNotificationAction} from "@shared/stores/notification/shared-notification.action";
import {tap} from "rxjs/operators";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OrderEnum,
  Override,
  OverrideProperty,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-space-organizational-unit-list-routable",
  templateUrl: "../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.html",
  styleUrls: ["../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceOrganizationalUnitListRoutable extends SharedAbstractListRoutable<OrganizationalUnit, PreservationSpaceOrganizationalUnitStateModel> {
  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  readonly KEY_BACK_BUTTON: string | undefined = undefined;
  readonly KEY_PARAM_NAME: keyof OrganizationalUnit & string = "name";

  protected onlyMyOrgUnits: boolean = false;

  @OverrideProperty()
  stickyTopPosition: number = 0;

  constructor(protected readonly _store: Store,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              private readonly _translate: TranslateService,
              private readonly _securityService: SecurityService) {
    super(_store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.preservationSpace_organizationalUnit, preservationSpaceOrganizationalUnitActionNameSpace, {
      canCreate: false,
      canGoBack: false,
      listExtraButtons: [
        {
          color: "primary",
          icon: IconNameEnum.sendRequest,
          labelToTranslate: (current) => MARK_AS_TRANSLATABLE("organizationalUnit.button.askCreationOrgunit"),
          callback: () => this.requestCreationOrgUnit(),
          order: 40,
          dataTest: DataTestEnum.preservationSpaceOrgUnitButtonAskCreationOrgUnit,
        },
        {
          color: "primary",
          icon: null,
          labelToTranslate: (current) => MARK_AS_TRANSLATABLE("organizationalUnit.button.onlyMyOrgUnit"),
          callback: (model, buttonElementRef, checked) => this.showOnlyMyOrgUnit(checked),
          order: 40,
          isToggleButton: true,
          isToggleChecked: false,
        },
      ],
    }, LocalStateEnum.preservationSpace);
  }

  showOnlyMyOrgUnit(checked: boolean): void {
    if (checked) {
      //show only authorized org units
      this._store.dispatch(new PreservationSpaceOrganizationalUnitAction.LoadOnlyMyOrgUnits());
      this.onlyMyOrgUnits = true;
      this.defineColumnsNoSortingNoFiltering();
    } else {
      // all org units
      this._store.dispatch(new PreservationSpaceOrganizationalUnitAction.GetAll());
      this.onlyMyOrgUnits = false;
      this.defineColumns();
    }
    this._changeDetector.detectChanges();
  }

  @Override()
  onQueryParametersEvent(queryParameters: QueryParameters): void {
    if (this.onlyMyOrgUnits) {
      this._store.dispatch(new PreservationSpaceOrganizationalUnitAction.LoadOnlyMyOrgUnits(queryParameters));
    } else {
      this._store.dispatch(new PreservationSpaceOrganizationalUnitAction.ChangeQueryParameters(queryParameters, true));
    }
    this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  requestCreationOrgUnit(): void {
    this.subscribe(this._dialog.open(PreservationSpaceOrganizationalUnitRequestCreationDialog, {
      minWidth: "500px",
    }).afterClosed().pipe(
      tap((result: OrgunitRequestCreationDialogResult | undefined) => {
        if (isNullOrUndefined(result)) {
          return;
        }
        this._sendRequestCreationOrgUnit(result);
      }),
    ));
  }

  private _sendRequestCreationOrgUnit(result: OrgunitRequestCreationDialogResult): void {
    this._store.dispatch(new SharedNotificationAction.Create({
      model: {
        emitter: MemoizedUtil.currentSnapshot(this._store, AppUserState),
        message: result.message,
        objectId: result.orgUnitName,
        notificationType: {
          resId: Enums.Notification.TypeEnum.CREATE_ORGUNIT_REQUEST,
        },
      },
    }));
  }

  protected defineActions(): DataTableActions<OrganizationalUnit>[] {
    return [
      {
        logo: IconNameEnum.edit,
        callback: model => this.goToEdit(model),
        placeholder: current => MARK_AS_TRANSLATABLE("crud.list.action.goToEdit"),
        displayOnCondition: model => this.conditionDisplayEditButton(model),
        isWrapped: false,
      },
      {
        logo: IconNameEnum.deposit,
        callback: model => this.goToDeposit(model),
        placeholder: current => MARK_AS_TRANSLATABLE("crud.list.action.goToDeposit"),
        displayOnCondition: model => this._securityService.isRootOrAdmin() || this._securityService.isMemberOfOrgUnit(model.resId),
        isWrapped: true,
      },
      {
        logo: IconNameEnum.sendRequest,
        callback: model => this.requestToBeMember(model.resId),
        placeholder: current => MARK_AS_TRANSLATABLE("organizationalUnit.requestToBeMember"),
        displayOnCondition: model => !this._securityService.isRootOrAdmin() && !this._securityService.isMemberOfOrgUnit(model.resId),
        isWrapped: true,
      },
      {
        logo: IconNameEnum.archiveAcl,
        callback: model => this.goToManageArchiveAcl(model),
        placeholder: current => LabelTranslateEnum.manageArchiveAcl,
        displayOnCondition: model => this._securityService.isStewardOfOrgUnit(model.resId),
        isWrapped: true,
      },
    ];
  }

  conditionDisplayEditButton(model: OrganizationalUnit | undefined): boolean {
    if (isNullOrUndefined(model)) {
      return true;
    }
    return this._securityService.isRootOrAdmin();
  }

  conditionDisplayDeleteButton(model: OrganizationalUnit | undefined): boolean {
    return false;
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "name",
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.ascending,
        isFilterable: true,
        isSortable: true,
        dataTest: DataTestEnum.preservationSpaceOrgUnitListSearchName,
      },
      {
        field: "openingDate",
        header: LabelTranslateEnum.opening,
        type: FieldTypeEnum.date,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "closingDate" as any,
        header: LabelTranslateEnum.closing,
        type: FieldTypeEnum.date,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "resId",
        header: "",
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        component: DataTableComponentEnum.organizationalUnitMember,
        isFilterable: false,
        isSortable: false,
        width: "40px",
      },
    ];
  }

  defineColumnsNoSortingNoFiltering(): void {
    this.columns = [
      {
        field: "name",
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.ascending,
        isFilterable: false,
        isSortable: false,
        dataTest: DataTestEnum.preservationSpaceOrgUnitListSearchName,
      },
      {
        field: "openingDate",
        header: LabelTranslateEnum.opening,
        type: FieldTypeEnum.date,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: false,
      },
      {
        field: "closingDate" as any,
        header: LabelTranslateEnum.closing,
        type: FieldTypeEnum.date,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: false,
      },
      {
        field: "resId",
        header: "",
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        component: DataTableComponentEnum.organizationalUnitMember,
        isFilterable: false,
        isSortable: false,
      },
    ];
  }

  goToDeposit(orgUnit: OrganizationalUnit): void {
    this._store.dispatch(new Navigate([RoutesEnum.deposit, orgUnit.resId]));
  }

  requestToBeMember(orgUnitResId: string): void {
    this.subscribe(this._dialog.open(PreservationSpaceOrganizationalUnitRequestAccessDialog, {
      minWidth: "500px",
    }).afterClosed().pipe(
      tap((result: OrgunitRequestAccessDialogResult | undefined) => {
        if (isNullOrUndefined(result)) {
          return;
        }
        this._sendRequestToBeMember(orgUnitResId, result);
      }),
    ));
  }

  goToManageArchiveAcl(orgUnit: OrganizationalUnit): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, orgUnit.resId, PreservationSpaceOrganizationalUnitRoutesEnum.manageAcl]));
  }

  private _sendRequestToBeMember(orgUnitResId: string, result: OrgunitRequestAccessDialogResult): void {
    this._store.dispatch(new SharedNotificationAction.Create({
      model: {
        emitter: MemoizedUtil.currentSnapshot(this._store, AppUserState),
        notifiedOrgUnit: {
          resId: orgUnitResId,
        },
        objectId: result.roleId,
        message: result.message,
        notificationType: {
          resId: Enums.Notification.TypeEnum.JOIN_ORGUNIT_REQUEST,
        },
      },
    }));
  }
}
