import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {environment} from "@environments/environment";
import {Archive} from "@home/models/archive.model";
import {
  ArchiveACL,
  User,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {PreservationSpaceOrganizationalUnitArchiveAclAction} from "@preservation-space/organizational-unit/stores/archive-acl/preservation-space-organizational-unit-archive-acl.action";
import {PreservationSpaceOrganizationalUnitArchiveAclState} from "@preservation-space/organizational-unit/stores/archive-acl/preservation-space-organizational-unit-archive-acl.state";
import {PreservationSpaceOrganizationalUnitAction} from "@preservation-space/organizational-unit/stores/preservation-space-organizational-unit.action";
import {SharedDeleteDialog} from "@shared/components/dialogs/shared-abstract-delete/shared-delete.dialog";
import {SharedDataTablePresentational} from "@shared/components/presentationals/shared-data-table/shared-data-table.presentational";
import {SharedAbstractListRoutableOption} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationSpaceOrganizationalUnitRoutesEnum,
  PreservationSpaceRoutesEnum,
  RoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {SharedArchiveAclAction} from "@shared/stores/archive-acl/shared-archive-acl.action";
import {sharedArchiveActionNameSpace} from "@shared/stores/archive/shared-archive.action";
import {
  PARAM_QUERY_ORGUNIT,
  PARAM_QUERY_SEARCH,
  SharedArchiveState,
} from "@shared/stores/archive/shared-archive.state";
import {sharedUserActionNameSpace} from "@shared/stores/user/shared-user.action";
import {SharedUserState} from "@shared/stores/user/shared-user.state";
import {StoreDialogUtil} from "@shared/utils/store-dialog.util";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
  QueryParametersUtil,
  ResourceNameSpace,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-space-organizational-unit-acl-list-routable",
  templateUrl: "../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.html",
  styleUrls: ["../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceOrganizationalUnitArchiveAclListRoutable extends SharedAbstractRoutable implements OnInit {
  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  readonly KEY_BACK_BUTTON: string | undefined = LabelTranslateEnum.backToOrganizationalUnit;
  readonly KEY_PARAM_NAME: keyof ArchiveACL & string = "aipId";

  skipInitialQuery: boolean = false;
  stickyTopPosition: number = environment.defaultStickyDatatableHeight;
  columnsSkippedToClear: string[] = [];

  actions: DataTableActions<ArchiveACL>[] = [
    {
      logo: IconNameEnum.edit,
      callback: (model: ArchiveACL) => this.goToEdit(model),
      placeholder: current => LabelTranslateEnum.edit,
      displayOnCondition: (model: ArchiveACL) => this.conditionDisplayEditButton(model),
    },
    {
      logo: IconNameEnum.delete,
      callback: (model: ArchiveACL) => this.delete(model),
      placeholder: current => LabelTranslateEnum.delete,
      displayOnCondition: (model: ArchiveACL) => this.conditionDisplayDeleteButton(model),
    },
  ];
  listNewId: string[];

  @ViewChild("dataTablePresentational")
  readonly dataTablePresentational: SharedDataTablePresentational<ArchiveACL>;

  sharedArchiveNameSpace: ResourceNameSpace = sharedArchiveActionNameSpace;
  sharedArchiveState: typeof SharedArchiveState = SharedArchiveState;

  sharedUserSort: Sort<User> = {
    field: "lastName",
    order: OrderEnum.ascending,
  };
  sharedUserNameSpace: ResourceNameSpace = sharedUserActionNameSpace;
  sharedUserState: typeof SharedUserState = SharedUserState;
  extraQueryParameters: MappingObject;

  columns: DataTableColumns<ArchiveACL>[];

  options: SharedAbstractListRoutableOption<ArchiveACL> = {
    canCreate: true,
    canGoBack: true,
    canRefresh: true,
  };

  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationSpaceOrganizationalUnitArchiveAclState);
  listObs: Observable<ArchiveACL[]> = MemoizedUtil.list(this._store, PreservationSpaceOrganizationalUnitArchiveAclState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, PreservationSpaceOrganizationalUnitArchiveAclState);
  private _orgUnitId: string;

  constructor(private readonly _store: Store,
              private readonly _changeDetector: ChangeDetectorRef,
              private readonly _route: ActivatedRoute,
              private readonly _routerExt: RouterExtService,
              private readonly _actions$: Actions,
              private readonly _dialog: MatDialog) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.retrieveResIdFromUrl();
    this._store.dispatch(new PreservationSpaceOrganizationalUnitAction.GetById(this._orgUnitId));

    this.extraQueryParameters = {[PARAM_QUERY_ORGUNIT]: this._orgUnitId};
    this.columns = [
      {
        field: "aipId",
        header: LabelTranslateEnum.archive,
        type: FieldTypeEnum.searchableSingleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        resourceNameSpace: this.sharedArchiveNameSpace,
        resourceState: this.sharedArchiveState as any,
        resourceLabelCallback: (value: Archive) => value.title,
        resourceLabelKey: PARAM_QUERY_SEARCH,
        component: DataTableComponentEnum.archive,
        extraSearchQueryParam: this.extraQueryParameters,
      },
      {
        field: "user.person.fullName",
        header: LabelTranslateEnum.user,
        type: FieldTypeEnum.searchableSingleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        resourceNameSpace: this.sharedUserNameSpace,
        resourceState: this.sharedUserState as any,
        resourceLabelKey: "lastName",
        resourceLabelCallback: (value: User) => value.lastName + ", " + value.firstName,
        searchableSingleSelectSort: this.sharedUserSort,
        filterableField: "user.resId",
        sortableField: "user.resId",
      },
      {
        field: "expiration",
        header: LabelTranslateEnum.expiration,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
    ];
  }

  protected retrieveResIdFromUrl(): void {
    this._orgUnitId = this._route.snapshot.parent.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
  }

  conditionDisplayEditButton(model: ArchiveACL | undefined): boolean {
    return true;
  }

  conditionDisplayDeleteButton(model: ArchiveACL | undefined): boolean {
    return true;
  }

  goToEdit(model: ArchiveACL): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, this._orgUnitId, PreservationSpaceOrganizationalUnitRoutesEnum.manageAcl, PreservationSpaceOrganizationalUnitRoutesEnum.manageAclDetail, model.resId, PreservationSpaceOrganizationalUnitRoutesEnum.manageAclEdit], {}, {skipLocationChange: true}));
  }

  delete(model: ArchiveACL): void {
    const datas = StoreDialogUtil.deleteData(LocalStateEnum.preservationSpace_organizationalUnit_archiveAcl);
    datas.name = model[this.KEY_PARAM_NAME]?.toString();
    datas.resId = model.resId;

    this._dialog.open(SharedDeleteDialog, {
      width: "400px",
      data: datas,
    });
    this.subscribe(this._actions$.pipe(ofActionCompleted(SharedArchiveAclAction.DeleteSuccess))
      .pipe(
        tap(result => {
          if (result.result.successful) {
            this.getAll();
          }
        }),
      ));
  }

  create(element: ElementRef): void {
    this._store.dispatch(new Navigate([
      AppRoutesEnum.preservationSpace,
      PreservationSpaceRoutesEnum.organizationalUnit,
      PreservationSpaceOrganizationalUnitRoutesEnum.detail,
      this._orgUnitId,
      PreservationSpaceOrganizationalUnitRoutesEnum.manageAcl,
      PreservationSpaceOrganizationalUnitRoutesEnum.manageAclCreate], {} /*{skipLocationChange: true}*/));
  }

  getAll(queryParameters?: QueryParameters): void {
    this._store.dispatch(new PreservationSpaceOrganizationalUnitArchiveAclAction.GetAll(queryParameters, true));
  }

  showDetail(model: ArchiveACL): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, this._orgUnitId, PreservationSpaceOrganizationalUnitRoutesEnum.manageAcl, PreservationSpaceOrganizationalUnitRoutesEnum.manageAclDetail, model.resId]));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    if (isNullOrUndefined(queryParameters)) {
      queryParameters = new QueryParameters();
    }
    const searchItem = QueryParametersUtil.getSearchItems(queryParameters);
    MappingObjectUtil.set(searchItem, "organizationalUnit.resId", this._orgUnitId);
    this._store.dispatch(new PreservationSpaceOrganizationalUnitArchiveAclAction.ChangeQueryParameters(queryParameters, true));
    this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  back(): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, this._orgUnitId]));
  }

  navigateNavigate(navigate: Navigate): void {
    this._store.dispatch(navigate);
  }
}
