import {PersonRole} from "@admin/models/person-role.model";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {NotificationHelper} from "@app/features/preservation-space/notification/helper/notification.helper";
import {environment} from "@environments/environment";
import {
  DisseminationPolicy,
  Institution,
  OrganizationalUnit,
  PreservationPolicy,
  ResearchDomain,
  Role,
  SubmissionPolicy,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {PreservationSpaceOrganizationalUnitAction} from "@preservation-space/organizational-unit/stores/preservation-space-organizational-unit.action";
import {PreservationSpaceOrganizationalUnitState} from "@preservation-space/organizational-unit/stores/preservation-space-organizational-unit.state";
import {PersonOrgUnitRoleMode} from "@shared/components/containers/shared-table-person-orgunit-role/shared-table-person-orgunit-role.container";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  PreservationSpaceOrganizationalUnitRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {SecurityService} from "@shared/services/security.service";
import {sharedInstitutionActionNameSpace} from "@shared/stores/institution/shared-institution.action";
import {SharedInstitutionState} from "@shared/stores/institution/shared-institution.state";
import {sharedResearchDomainActionNameSpace} from "@shared/stores/research-domain/shared-research-domain.action";
import {SharedResearchDomainState} from "@shared/stores/research-domain/shared-research-domain.state";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  DateUtil,
  isEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ObservableUtil,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-space-organizational-unit-form",
  templateUrl: "./preservation-space-organizational-unit-form.presentational.html",
  styleUrls: ["./preservation-space-organizational-unit-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceOrganizationalUnitFormPresentational extends SharedAbstractFormPresentational<OrganizationalUnit> implements OnInit {
  private readonly KEY_SOURCE: keyof ResearchDomain = "source";
  private readonly KEY_DEFAULT: string = "default";
  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnInput;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  organizationalUnitActionNameSpace: ResourceLogoNameSpace = PreservationSpaceOrganizationalUnitAction;
  organizationalUnitState: typeof PreservationSpaceOrganizationalUnitState = PreservationSpaceOrganizationalUnitState;

  @Input()
  researchDomainSources: string[];

  @Input()
  listSubmissionPolicies: SubmissionPolicy[];

  @Input()
  listPreservationPolicies: PreservationPolicy[];

  @Input()
  listDisseminationPolicies: DisseminationPolicy[];

  @Input()
  selectedInstitutions: Institution[];

  private _selectedSubmissionPolicies: SubmissionPolicy[];

  @Input()
  set selectedSubmissionPolicies(value: SubmissionPolicy[]) {
    this._selectedSubmissionPolicies = this._setDefaultPolicyInfo(value, this.model?.defaultSubmissionPolicy?.resId);
  }

  get selectedSubmissionPolicies(): SubmissionPolicy[] {
    return this._selectedSubmissionPolicies;
  }

  private _selectedPreservationPolicies: PreservationPolicy[];

  @Input()
  set selectedPreservationPolicies(value: PreservationPolicy[]) {
    this._selectedPreservationPolicies = this._setDefaultPolicyInfo(value, this.model?.defaultPreservationPolicy?.resId);
  }

  get selectedPreservationPolicies(): PreservationPolicy[] {
    return this._selectedPreservationPolicies;
  }

  private _selectedDisseminationPolicies: DisseminationPolicy[];

  @Input()
  set selectedDisseminationPolicies(value: DisseminationPolicy[]) {
    this._selectedDisseminationPolicies = this._setDefaultPolicyInfo(value, this.model?.defaultDisseminationPolicy?.resId);
  }

  get selectedDisseminationPolicies(): DisseminationPolicy[] {
    return this._selectedDisseminationPolicies;
  }

  @Input()
  selectedPersonRole: PersonRole[];

  @Input()
  listRole: Role[];

  @Input()
  isManager: boolean;

  @Input()
  urlQueryParameters: MappingObject<string | undefined>;

  @Input()
  isTour: boolean;

  researchDomainLabelCallback: (value: ResearchDomain) => string = (value: ResearchDomain) => `[${value.source}] ${value.name}`;

  institutionLabelCallback: (value: Institution) => string = (value: Institution) => `${value.name}`;

  protected readonly _requestToBeMemberBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("requestToBeMemberChange")
  readonly requestToBeMemberObs: Observable<void> = ObservableUtil.asObservable(this._requestToBeMemberBS);

  protected readonly _seeArchiveBS: BehaviorSubject<OrganizationalUnit> = new BehaviorSubject<OrganizationalUnit>(undefined);
  @Output("seeArchiveChange")
  readonly seeArchiveObs: Observable<OrganizationalUnit> = ObservableUtil.asObservable(this._seeArchiveBS);

  sharedInstitutionSort: Sort<Institution> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedInstitutionActionNameSpace: ResourceNameSpace = sharedInstitutionActionNameSpace;
  sharedInstitutionState: typeof SharedInstitutionState = SharedInstitutionState;

  sharedResearchDomainSort: Sort<ResearchDomain> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedResearchDomainActionNameSpace: ResourceNameSpace = sharedResearchDomainActionNameSpace;
  sharedResearchDomainState: typeof SharedResearchDomainState = SharedResearchDomainState;

  extraSearchParameterSource: MappingObject = {};
  personIdAutomaticallyAdded: string | undefined = undefined;
  columnsPreservationPolicies: DataTableColumns<PreservationPolicy>[] = [
    {
      field: "name",
      header: LabelTranslateEnum.preservationPolicies,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: "smartRetention",
      header: LabelTranslateEnum.retention,
      type: FieldTypeEnum.number,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: "dispositionApproval",
      header: LabelTranslateEnum.disposalApproval,
      type: FieldTypeEnum.boolean,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: this.KEY_DEFAULT,
      header: LabelTranslateEnum.defaultValue,
      type: FieldTypeEnum.boolean,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
  ];
  columnsDisseminationPolicies: DataTableColumns<DisseminationPolicy>[] = [
    {
      field: "name",
      header: LabelTranslateEnum.disseminationPolicy,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: this.KEY_DEFAULT,
      header: LabelTranslateEnum.defaultValue,
      type: FieldTypeEnum.boolean,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
  ];
  columnsSubmissionPolicies: DataTableColumns<SubmissionPolicy>[] = [
    {
      field: "name",
      header: LabelTranslateEnum.submissionPolicy,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: "timeToKeep",
      header: LabelTranslateEnum.timeToKeep,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: this.KEY_DEFAULT,
      header: LabelTranslateEnum.defaultValue,
      type: FieldTypeEnum.boolean,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
  ];

  get personOrgUnitRoleMode(): typeof PersonOrgUnitRoleMode {
    return PersonOrgUnitRoleMode;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly breakpointService: BreakpointService,
              private readonly _notificationService: NotificationService,
              public readonly securityService: SecurityService) {
    super(_changeDetectorRef, _elementRef);
  }

  ngOnInit(): void {
    if (this.isTour) {
      this.bindFormTo({
        name: "Tour organizational unit",
      });
      return;
    }
    super.ngOnInit();
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [SolidifyValidator]],
      [this.formDefinition.url]: ["", [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.openingDate]: [""],
      [this.formDefinition.closingDate]: [""],
      [this.formDefinition.personRole]: ["", [SolidifyValidator]],
      [this.formDefinition.filterResearchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.researchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.keywords]: [[], [SolidifyValidator]],
      [this.formDefinition.institutions]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(organizationalUnit: OrganizationalUnit): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [organizationalUnit.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [organizationalUnit.description, [SolidifyValidator]],
      [this.formDefinition.url]: [organizationalUnit.url, [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.openingDate]: [organizationalUnit.openingDate],
      [this.formDefinition.closingDate]: [organizationalUnit.closingDate],
      [this.formDefinition.personRole]: ["", [SolidifyValidator]],
      [this.formDefinition.filterResearchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.researchDomains]: [organizationalUnit.researchDomains?.map(r => r.resId), [SolidifyValidator]],
      [this.formDefinition.keywords]: [isNullOrUndefined(organizationalUnit.keywords) ? [] : [...organizationalUnit.keywords], [SolidifyValidator]],
      [this.formDefinition.institutions]: [this.selectedInstitutions?.map(r => r.resId), [SolidifyValidator]],
    });

    this._addRoleAskingInNotificationRequest();
  }

  private _addRoleAskingInNotificationRequest(): void {
    if (isNullOrUndefined(this.urlQueryParameters) || MappingObjectUtil.size(this.urlQueryParameters) === 0) {
      return;
    }
    const roleId = MappingObjectUtil.get(this.urlQueryParameters, NotificationHelper.KEY_ROLE_ID);
    const personId = MappingObjectUtil.get(this.urlQueryParameters, NotificationHelper.KEY_PERSON_ID);
    if (isNullOrUndefined(roleId) || isNullOrUndefined(personId)) {
      return;
    }
    this.personIdAutomaticallyAdded = personId;
    const existingPerson = this.selectedPersonRole.find(p => p.resId === personId);
    if (isNullOrUndefined(existingPerson)) {
      this.selectedPersonRole = [...this.selectedPersonRole, {
        resId: personId,
        roles: [{
          resId: roleId,
        }],
      }];
      this.form.markAsDirty();
      this._notificationService.showInformation(MARK_AS_TRANSLATABLE("organizationalUnit.notification.personAddedToOrgUnit"));
    } else {
      this._notificationService.showInformation(MARK_AS_TRANSLATABLE("organizationalUnit.notification.personAlreadyInOrgUnit"));
    }
  }

  protected treatmentBeforeSubmit(organizationalUnit: OrganizationalUnit): OrganizationalUnit {
    delete organizationalUnit[this.formDefinition.filterResearchDomains];
    organizationalUnit.openingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.openingDate);
    organizationalUnit.closingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.closingDate);
    organizationalUnit.researchDomains = [];
    if (this.form.get(this.formDefinition.researchDomains).value !== "" && isNotNullNorUndefined(this.form.get(this.formDefinition.researchDomains).value)) {
      this.form.get(this.formDefinition.researchDomains).value.forEach(resId => {
        organizationalUnit.researchDomains.push({resId: resId});
      });
    }
    organizationalUnit.institutions = [];
    if (this.form.get(this.formDefinition.institutions).value !== "" && isNotNullNorUndefined(this.form.get(this.formDefinition.institutions).value)) {
      this.form.get(this.formDefinition.institutions).value.forEach(resId => {
        organizationalUnit.institutions.push({resId: resId});
      });
    }
    return organizationalUnit;
  }

  private _setDefaultPolicyInfo(list: any[], defaultPolicyResId: string | undefined): any[] {
    if (isNullOrUndefined(list)) {
      return [];
    }
    list = [...list];
    const defaultValue = defaultPolicyResId;
    if (isNullOrUndefined(defaultValue)) {
      return list;
    }
    const defaultItemIndex = list.findIndex(i => i.resId === defaultValue);
    if (defaultItemIndex === -1) {
      return list;
    }
    const defaultItem = {...list[defaultItemIndex]};
    defaultItem[this.KEY_DEFAULT] = true;
    list[defaultItemIndex] = defaultItem;
    return list;
  }

  requestToBeMember(): void {
    this._requestToBeMemberBS.next();
  }

  goToReceivedRequestNotification(): void {
    this._navigateWithQueryParamBS.next(new Navigate([RoutesEnum.preservationSpaceNotificationInbox], NotificationHelper.getUrlQueryParam(this.model.resId)));
  }

  goToSentRequestNotification(): void {
    this._navigateWithQueryParamBS.next(new Navigate([RoutesEnum.preservationSpaceNotificationSent], NotificationHelper.getUrlQueryParam(this.model.resId)));
  }

  goToDeposit(): void {
    this._navigateBS.next([RoutesEnum.deposit, this.model.resId]);
  }

  goToArchive(): void {
    this._seeArchiveBS.next(this.model);
  }

  goToManageArchiveAcl(): void {
    this._navigateBS.next([RoutesEnum.preservationSpaceOrganizationalUnitDetail, this.model.resId, PreservationSpaceOrganizationalUnitRoutesEnum.manageAcl]);
  }

  filterResearchDomain(source: string): void {
    this._computeExtraSearchParameterCategory(source);
  }

  private _computeExtraSearchParameterCategory(value: string): void {
    if (isNullOrUndefined(value) || isEmptyString(value)) {
      MappingObjectUtil.delete(this.extraSearchParameterSource, this.KEY_SOURCE as string);
      return;
    }
    MappingObjectUtil.set(this.extraSearchParameterSource, this.KEY_SOURCE as string, value);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() url: string;
  @PropertyName() closingDate: string;
  @PropertyName() openingDate: string;
  @PropertyName() personRole: string;
  @PropertyName() filterResearchDomains: string;
  @PropertyName() researchDomains: string;
  @PropertyName() keywords: string;
  @PropertyName() preservationPolicy: string;
  @PropertyName() institutions: string;
}
