import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  Validators,
} from "@angular/forms";
import {ActivatedRoute} from "@angular/router";
import {
  ArchiveACL,
  User,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {sharedArchiveActionNameSpace} from "@shared/stores/archive/shared-archive.action";
import {
  PARAM_QUERY_ORGUNIT,
  PARAM_QUERY_SEARCH,
  SharedArchiveState,
} from "@shared/stores/archive/shared-archive.state";
import {sharedUserActionNameSpace} from "@shared/stores/user/shared-user.action";
import {SharedUserState} from "@shared/stores/user/shared-user.state";
import {Observable} from "rxjs";
import {
  filter,
  tap,
} from "rxjs/operators";
import {
  DateUtil,
  isTruthyObject,
  MappingObject,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-space-organizational-unit-archive-acl-form",
  templateUrl: "./preservation-space-organizational-unit-archive-acl-form.presentational.html",
  styleUrls: ["./preservation-space-organizational-unit-archive-acl-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceOrganizationalUnitArchiveAclFormPresentational extends SharedAbstractFormPresentational<ArchiveACL> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  sharedArchiveNameSpace: ResourceNameSpace = sharedArchiveActionNameSpace;
  sharedArchiveState: typeof SharedArchiveState = SharedArchiveState;

  @Input()
  orgUnitId: string;

  extraQueryParameters: MappingObject;
  searchKey: string = PARAM_QUERY_SEARCH;

  sharedUserSort: Sort<User> = {
    field: "lastName",
    order: OrderEnum.ascending,
  };
  sharedUserNameSpace: ResourceNameSpace = sharedUserActionNameSpace;
  sharedUserState: typeof SharedUserState = SharedUserState;
  labelUserCallback: (user: User) => string = user => user.lastName + ", " + user.firstName;

  aipFormControl: AbstractControl;
  userFormControl: AbstractControl;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              protected readonly _route: ActivatedRoute,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.extraQueryParameters = {[PARAM_QUERY_ORGUNIT]: this.orgUnitId};
    this.aipFormControl = this.form.get(this.formDefinition.aipId);
    this.userFormControl = this.form.get(this.formDefinition.user);

    this.subscribe(this._observableRemoveErrorField(this.aipFormControl));
    this.subscribe(this._observableRemoveErrorField(this.userFormControl));
  }

  private _observableRemoveErrorField(formControl: AbstractControl): Observable<string> {
    return formControl.valueChanges.pipe(
      filter(() => isTruthyObject(formControl.errors)),
      tap(() => {
        this._removeErrorOnAipAndUserField();
      }),
    );
  }

  private _removeErrorOnAipAndUserField(): void {
    this._removeErrorOnFormControl(this.aipFormControl);
    this._removeErrorOnFormControl(this.userFormControl);

    this._changeDetectorRef.detectChanges();
  }

  private _removeErrorOnFormControl(formControl: AbstractControl): void {
    formControl.setErrors(null);
    formControl.markAsTouched();
    formControl.updateValueAndValidity();
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.aipId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.user]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.expiration]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(archiveAcl: ArchiveACL): void {
    this.form = this._fb.group({
      [this.formDefinition.aipId]: [archiveAcl.aipId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.user]: [archiveAcl.user.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.expiration]: [DateUtil.convertOffsetDateTimeIso8601ToDate(archiveAcl.expiration), [SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(archiveAcl: ArchiveACL): ArchiveACL {
    const archiveAclToSubmit = {
      resId: this.model?.resId,
      aipId: this.form.get(this.formDefinition.aipId).value,
      organizationalUnit: {
        resId: this.orgUnitId,
      },
      user: {
        resId: this.form.get(this.formDefinition.user).value,
      },
      expiration: DateUtil.convertToOffsetDateTimeIso8601(this.form.get(this.formDefinition.expiration).value),
    } as ArchiveACL;
    return archiveAclToSubmit;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() aipId: string;
  @PropertyName() organizationalUnitId: string;
  @PropertyName() user: string;
  @PropertyName() expiration: string;
}
