import {Injectable} from "@angular/core";
import {NotificationModeEnum} from "@app/features/preservation-space/notification/enums/notification-mode.enum";
import {
  PreservationSpaceNotificationAction,
  preservationSpaceNotificationActionNameSpace,
} from "@app/features/preservation-space/notification/stores/preservation-space-notification.action";
import {
  PreservationSpaceNotificationStatusHistoryState,
  PreservationSpaceNotificationStatusHistoryStateModel,
} from "@app/features/preservation-space/notification/stores/status-history/preservation-space-notification-status-history.state";
import {AppAction} from "@app/stores/app.action";
import {Enums} from "@enums";
import {NotificationDlcm} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";
import {ViewModeEnum} from "@shared/enums/view-mode.enum";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {Result} from "@shared/models/business/result.model";
import {SharedRoleAction} from "@shared/stores/role/shared-role.action";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {
  Observable,
  pipe,
} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  QueryParametersUtil,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface PreservationSpaceNotificationStateModel extends ResourceStateModel<NotificationDlcm> {
  mode: NotificationModeEnum;
  orgUnitId: string | undefined;
  preservationSpace_notification_statusHistory: PreservationSpaceNotificationStatusHistoryStateModel;
}

@Injectable()
@State<PreservationSpaceNotificationStateModel>({
  name: LocalStateEnum.preservationSpace_notification,
  defaults: {
    ...defaultResourceStateInitValue(),
    mode: NotificationModeEnum.inbox,
    orgUnitId: undefined,
    preservationSpace_notification_statusHistory: {...defaultStatusHistoryInitValue()},
  },
  children: [
    PreservationSpaceNotificationStatusHistoryState,
  ],
})
export class PreservationSpaceNotificationState extends ResourceState<PreservationSpaceNotificationStateModel, NotificationDlcm> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationSpaceNotificationActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.preservationSpaceNotificationInboxDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.preservationSpaceNotificationInboxDetail + urlSeparator + resId,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservationSpace.notifications.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservationSpace.notifications.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservationSpace.notifications.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.notifications;
  }

  @Selector()
  static isLoading(state: PreservationSpaceNotificationStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: PreservationSpaceNotificationStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: PreservationSpaceNotificationStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.emitter["person"].fullName;
  }

  @Selector()
  static isReadyToBeDisplayed(state: PreservationSpaceNotificationStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: PreservationSpaceNotificationStateModel): boolean {
    return true;
  }

  @OverrideDefaultAction()
  @Action(PreservationSpaceNotificationAction.LoadResource)
  loadResource(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedRoleAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new PreservationSpaceNotificationAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new PreservationSpaceNotificationAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @Action(PreservationSpaceNotificationAction.SetMode)
  setMode(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetMode): void {
    ctx.patchState({
      mode: action.mode,
      orgUnitId: action.orgUnitId,
    });
  }

  @OverrideDefaultAction()
  @Action(PreservationSpaceNotificationAction.GetAll)
  getAll(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.GetAll): Observable<CollectionTyped<NotificationDlcm>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        list: undefined,
        total: 0,
      };
    }
    let queryParameters = StoreUtil.getQueryParametersToApply(action.queryParameters, ctx);
    const orgUnitId = ctx.getState().orgUnitId;
    if (!isNullOrUndefined(orgUnitId)) {
      queryParameters = QueryParametersUtil.clone(queryParameters);
      const search = QueryParametersUtil.getSearchItems(queryParameters);
      MappingObjectUtil.set(search, "notifiedOrgUnit.resId", orgUnitId);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: queryParameters,
      ...reset,
    });
    let mode = ApiActionEnum.INBOX;
    if (ctx.getState().mode === NotificationModeEnum.sent) {
      mode = ApiActionEnum.SENT;
    }
    return this.apiService.get<NotificationDlcm>(this._urlResource + urlSeparator + mode, ctx.getState().queryParameters)
      .pipe(
        action.cancelIncomplete ? StoreUtil.cancelUncompleted(ctx, this.actions$, [this._nameSpace.GetAll, Navigate]) : pipe(),
        tap((collection: CollectionTyped<NotificationDlcm>) => {
          ctx.dispatch(new PreservationSpaceNotificationAction.GetAllSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new PreservationSpaceNotificationAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(PreservationSpaceNotificationAction.GetById)
  getById(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.GetById): Observable<NotificationDlcm> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    let mode = ApiActionEnum.INBOX;
    if (ctx.getState().mode === NotificationModeEnum.sent) {
      mode = ApiActionEnum.SENT;
    }
    return this.apiService.getById<NotificationDlcm>(this._urlResource + urlSeparator + mode, action.id)
      .pipe(
        tap((model: NotificationDlcm) => {
          ctx.dispatch(new PreservationSpaceNotificationAction.GetByIdSuccess(action, model));
        }),
        catchError(error => {
          ctx.dispatch(new PreservationSpaceNotificationAction.GetByIdFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationSpaceNotificationAction.SetProcessed)
  setProcessed(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetProcessed): Observable<Result> {
    return this.apiService.post<Result>(this._urlResource + urlSeparator + action.notificationId + urlSeparator + ApiActionEnum.SET_PROCESSED)
      .pipe(
        tap(result => ctx.dispatch(new PreservationSpaceNotificationAction.SetProcessedSuccess(action))),
        catchError(error => {
          ctx.dispatch(new PreservationSpaceNotificationAction.SetProcessedFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationSpaceNotificationAction.SetProcessedSuccess)
  setProcessedSuccess(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetProcessedSuccess): void {
    this.removeNotificationPendingListAndRefresh(ctx, (action.parentAction as PreservationSpaceNotificationAction.SetRefuse).notificationId, (action.parentAction as PreservationSpaceNotificationAction.SetRefuse).mode);
    const notificationCategory = (action.parentAction as PreservationSpaceNotificationAction.SetProcessed).notificationCategory;
    if (notificationCategory === Enums.Notification.CategoryEnum.REQUEST) {
      this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setProcessed.success"));
    } else if (notificationCategory === Enums.Notification.CategoryEnum.INFO) {
      this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setRead.success"));
    }
    SessionStorageHelper.removeItemInList(SessionStorageEnum.notificationInboxPending, (action.parentAction as PreservationSpaceNotificationAction.SetProcessed).notificationId);
    this.store.dispatch(new AppAction.UpdateNotificationInbox);
  }

  @Action(PreservationSpaceNotificationAction.SetProcessedFail)
  setProcessedFail(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetProcessedFail): void {
    const notificationCategory = (action.parentAction as PreservationSpaceNotificationAction.SetProcessed).notificationCategory;
    if (notificationCategory === Enums.Notification.CategoryEnum.REQUEST) {
      this.notificationService.showError(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setProcessed.fail"));
    } else if (notificationCategory === Enums.Notification.CategoryEnum.INFO) {
      this.notificationService.showError(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setRead.fail"));
    }
  }

  @Action(PreservationSpaceNotificationAction.SetRefuse)
  setRefuse(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetRefuse): Observable<Result> {
    return this.apiService.post<Result>(this._urlResource + urlSeparator + action.notificationId + urlSeparator + ApiActionEnum.SET_REFUSED)
      .pipe(
        tap(result => ctx.dispatch(new PreservationSpaceNotificationAction.SetRefuseSuccess(action))),
        catchError(error => {
          ctx.dispatch(new PreservationSpaceNotificationAction.SetRefuseFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationSpaceNotificationAction.SetRefuseSuccess)
  setRefuseSuccess(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetRefuseSuccess): void {
    this.removeNotificationPendingListAndRefresh(ctx, (action.parentAction as PreservationSpaceNotificationAction.SetRefuse).notificationId, (action.parentAction as PreservationSpaceNotificationAction.SetRefuse).mode);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setRefuse.success"));
    SessionStorageHelper.removeItemInList(SessionStorageEnum.notificationInboxPending, (action.parentAction as PreservationSpaceNotificationAction.SetRefuse).notificationId);
    this.store.dispatch(new AppAction.UpdateNotificationInbox);
  }

  @Action(PreservationSpaceNotificationAction.SetRefuseFail)
  setRefuseFail(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetRefuseFail): void {
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setRefuse.fail"));
  }

  @Action(PreservationSpaceNotificationAction.SetPending)
  setPending(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetPending): Observable<Result> {
    return this.apiService.post<Result>(this._urlResource + urlSeparator + action.notificationId + urlSeparator + ApiActionEnum.SET_PENDING)
      .pipe(
        tap(result => ctx.dispatch(new PreservationSpaceNotificationAction.SetPendingSuccess(action))),
        catchError(error => {
          ctx.dispatch(new PreservationSpaceNotificationAction.SetPendingFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationSpaceNotificationAction.SetPendingSuccess)
  setPendingSuccess(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetPendingSuccess): void {
    this.removeNotificationPendingListAndRefresh(ctx, (action.parentAction as PreservationSpaceNotificationAction.SetPending).notificationId, (action.parentAction as PreservationSpaceNotificationAction.SetPending).mode);
    const notificationCategory = (action.parentAction as PreservationSpaceNotificationAction.SetPending).notificationCategory;
    if (notificationCategory === Enums.Notification.CategoryEnum.REQUEST) {
      this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setPending.success"));
    } else if (notificationCategory === Enums.Notification.CategoryEnum.INFO) {
      this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setUnread.success"));
    }
    SessionStorageHelper.addItemInList(SessionStorageEnum.notificationInboxPending, (action.parentAction as PreservationSpaceNotificationAction.SetPending).notificationId);
    this.store.dispatch(new AppAction.UpdateNotificationInbox);
  }

  private removeNotificationPendingListAndRefresh(ctx: StateContext<PreservationSpaceNotificationStateModel>,
                                                  notificationId: string,
                                                  mode: ViewModeEnum): void {
    SessionStorageHelper.removeItemInList(SessionStorageEnum.notificationInboxPending, notificationId);
    if (mode === ViewModeEnum.detail) {
      ctx.dispatch(new PreservationSpaceNotificationAction.GetById(notificationId));
    } else if (mode === ViewModeEnum.list) {
      ctx.dispatch(new PreservationSpaceNotificationAction.GetAll());
    }
  }

  @Action(PreservationSpaceNotificationAction.SetPendingFail)
  setPendingFail(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.SetPendingFail): void {
    const notificationCategory = (action.parentAction as PreservationSpaceNotificationAction.SetPending).notificationCategory;
    if (notificationCategory === Enums.Notification.CategoryEnum.REQUEST) {
      this.notificationService.showError(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setPending.fail"));
    } else if (notificationCategory === Enums.Notification.CategoryEnum.INFO) {
      this.notificationService.showError(MARK_AS_TRANSLATABLE("preservationSpace.notifications.setUnread.fail"));
    }
  }

  private redirectToNotificationDetail(ctx: StateContext<PreservationSpaceNotificationStateModel>, notificationId: string): void {
    // ctx.dispatch(new Navigate([RoutesEnum.preservationSpaceNotificationInboxDetail, notificationId], NotificationHelper.getUrlQueryParam(ctx.getState().orgUnitId)));
    ctx.dispatch(new PreservationSpaceNotificationAction.GetById(notificationId));
  }

  @OverrideDefaultAction()
  @Action(PreservationSpaceNotificationAction.Clean)
  clean(ctx: StateContext<PreservationSpaceNotificationStateModel>, action: PreservationSpaceNotificationAction.Clean): void {
    const orgUnitId = ctx.getState().orgUnitId;
    const mode = ctx.getState().mode;
    super.clean(ctx, action);
    ctx.patchState({
      orgUnitId,
      mode,
    });
  }
}
