import {ActivatedRoute} from "@angular/router";
import {NotificationModeEnum} from "@app/features/preservation-space/notification/enums/notification-mode.enum";
import {Enums} from "@enums";
import {NotificationDlcm} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {
  DepositRoutesEnum,
  PreservationSpaceOrganizationalUnitRoutesEnum,
  PreservationSpaceRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {
  isNotNullNorUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
} from "solidify-frontend";

export class NotificationHelper {
  static readonly KEY_ORGUNIT_ID: keyof NotificationDlcm = "notifiedOrgUnit";
  static readonly KEY_ROLE_ID: string = "roleId";
  static readonly KEY_PERSON_ID: string = "personId";
  static readonly KEY_ORGUNIT_NAME: string = "orgUnitName";

  static getUrlQueryParam(orgUnitId: string): object {
    if (isNotNullNorUndefined(orgUnitId)) {
      return {
        [this.KEY_ORGUNIT_ID]: orgUnitId,
      };
    }
    return {};
  }

  static getMode(route: ActivatedRoute): NotificationModeEnum {
    const url = route.snapshot.parent.url.toString();
    let mode = NotificationModeEnum.inbox;
    if (url === PreservationSpaceRoutesEnum.notificationSent) {
      mode = NotificationModeEnum.sent;
    }
    return mode;
  }

  static process(notification: NotificationDlcm, store: Store, notificationService: NotificationService): void {
    switch (notification.notificationType?.resId) {
      case Enums.Notification.TypeEnum.JOIN_ORGUNIT_REQUEST:
        this._processJoinOrgUnitRequest(notification, store);
        break;
      case Enums.Notification.TypeEnum.VALIDATE_DEPOSIT_REQUEST:
      case Enums.Notification.TypeEnum.IN_ERROR_DEPOSIT_INFO:
      case Enums.Notification.TypeEnum.CREATED_DEPOSIT_INFO:
        this._processDepositRequest(notification, store);
        break;
      case Enums.Notification.TypeEnum.CREATE_ORGUNIT_REQUEST:
        this._processCreateOrgUnitRequest(notification, store);
        break;
      case Enums.Notification.TypeEnum.IN_ERROR_AIP_INFO:
        this._processAipInError(notification, store);
        break;
      case Enums.Notification.TypeEnum.IN_ERROR_DOWNLOADED_AIP_INFO:
        this._processAipDownloadedInError(notification, store);
        break;
      case Enums.Notification.TypeEnum.IN_ERROR_DIP_INFO:
        this._processDipInError(notification, store);
        break;
      case Enums.Notification.TypeEnum.IN_ERROR_SIP_INFO:
        this._processSipInError(notification, store);
        break;
      case Enums.Notification.TypeEnum.ACCESS_DATASET_REQUEST:
        this._processAccessDataSetRequest(notification, store);
        break;
      case Enums.Notification.TypeEnum.APPROVE_DISPOSAL_BY_ORGUNIT_REQUEST:
        this._processApproveDisposalRequestByOrgUnit(notification, store);
        break;
      case Enums.Notification.TypeEnum.APPROVE_DISPOSAL_REQUEST:
        this._processApproveDisposalRequest(notification, store);
        break;
      default:
        notificationService.showInformation(MARK_AS_TRANSLATABLE("preservationSpace.notifications.notification.processActionNotAvailableForThisType"), {type: notification.notificationType?.resId});
        break;
    }
  }

  private static _processJoinOrgUnitRequest(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, notification.notifiedOrgUnit.resId, PreservationSpaceOrganizationalUnitRoutesEnum.edit], {
      [NotificationHelper.KEY_ROLE_ID]: notification.objectId, // contain roleId
      [NotificationHelper.KEY_PERSON_ID]: notification.emitter.person.resId,
    }, {
      skipLocationChange: false,
    }));
  }

  private static _processDepositRequest(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.deposit, notification.notifiedOrgUnit.resId, DepositRoutesEnum.detail, notification.objectId])); // contain deposit id
  }

  private static _processAipInError(notification: NotificationDlcm, store: Store): void {
    // TODO Use here default storagion number
    store.dispatch(new Navigate([RoutesEnum.preservationPlanningAip, 1, SharedAipRoutesEnum.aipDetail, notification.objectId])); // contain aip id
  }

  private static _processAipDownloadedInError(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationPlanningAipDownloadedDetail, notification.objectId])); // contain aip downloaded id
  }

  private static _processSipInError(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationPlanningSipDetail, notification.objectId])); // contain object id
  }

  private static _processDipInError(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationPlanningDipDetail, notification.objectId])); // contain dip id
  }

  private static _processCreateOrgUnitRequest(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.adminOrganizationalUnitCreate], {
      [NotificationHelper.KEY_ROLE_ID]: Enums.Role.RoleEnum.MANAGER,
      [NotificationHelper.KEY_PERSON_ID]: notification.emitter.person.resId,
      [NotificationHelper.KEY_ORGUNIT_NAME]: notification.objectId, // contain org unit name
    }, {
      skipLocationChange: false,
    }));
  }

  private static _processAccessDataSetRequest(notification: NotificationDlcm, store: Store): void {
    // TODO Manage ACL Case here if dataset is Closed
    store.dispatch(new Navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, notification.notifiedOrgUnit.resId, PreservationSpaceOrganizationalUnitRoutesEnum.edit], {
      [NotificationHelper.KEY_ROLE_ID]: Enums.Role.RoleEnum.VISITOR,
      [NotificationHelper.KEY_PERSON_ID]: notification.emitter.person.resId,
    }));
  }

  private static _processApproveDisposalRequestByOrgUnit(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationSpaceAipStewardDetail, notification.objectId]));
  }

  private static _processApproveDisposalRequest(notification: NotificationDlcm, store: Store): void {
    store.dispatch(new Navigate([RoutesEnum.preservationPlanningAip, 1, SharedAipRoutesEnum.aipDetail, notification.objectId]));
  }
}
