import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {NotificationModeEnum} from "@app/features/preservation-space/notification/enums/notification-mode.enum";
import {NotificationHelper} from "@app/features/preservation-space/notification/helper/notification.helper";
import {
  PreservationSpaceNotificationState,
  PreservationSpaceNotificationStateModel,
} from "@app/features/preservation-space/notification/stores/preservation-space-notification.state";
import {Enums} from "@enums";
import {
  NotificationDlcm,
  OrganizationalUnit,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractListRoutable} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  PreservationSpaceRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {ViewModeEnum} from "@shared/enums/view-mode.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {SecurityService} from "@shared/services/security.service";
import {sharedOrganizationalUnitActionNameSpace} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrganizationalUnitState} from "@shared/stores/organizational-unit/shared-organizational-unit.state";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {Observable} from "rxjs";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  OrderEnum,
  Override,
  ResourceNameSpace,
  Sort,
} from "solidify-frontend";
import {
  PreservationSpaceNotificationAction,
  preservationSpaceNotificationActionNameSpace,
} from "../../../stores/preservation-space-notification.action";

@Component({
  selector: "dlcm-preservation-space-notification-list-routable",
  templateUrl: "./preservation-space-notification-list.routable.html",
  styleUrls: ["../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceNotificationListRoutable extends SharedAbstractListRoutable<NotificationDlcm, PreservationSpaceNotificationStateModel> implements OnInit {
  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  readonly KEY_BACK_BUTTON: string | undefined = MARK_AS_TRANSLATABLE("preservationSpace.notifications.button.goBackToPreservationSpace");
  readonly KEY_PARAM_NAME: keyof NotificationDlcm & string = "notificationType";
  readonly KEY_QUERY_PARAMETER_NOTIFIED_ORGUNIT_ID: keyof NotificationDlcm | string = "notifiedOrgUnit.resId";

  protected _resId: string;

  orgUnitIdObs: Observable<string> = MemoizedUtil.select(this._store, PreservationSpaceNotificationState, state => state.orgUnitId);
  modeObs: Observable<NotificationModeEnum> = MemoizedUtil.select(this._store, PreservationSpaceNotificationState, state => state.mode);
  mode: NotificationModeEnum;

  sharedOrgUnitSort: Sort<OrganizationalUnit> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedOrgUnitActionNameSpace: ResourceNameSpace = sharedOrganizationalUnitActionNameSpace;
  sharedOrganizationalUnitState: typeof SharedOrganizationalUnitState = SharedOrganizationalUnitState;
  isHighlightCondition: (notification: NotificationDlcm) => boolean = notification => notification.notificationStatus === Enums.Notification.StatusEnum.PENDING;

  get notificationModeEnum(): typeof NotificationModeEnum {
    return NotificationModeEnum;
  }

  constructor(protected readonly _store: Store,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              private readonly _securityService: SecurityService,
              private readonly _notificationService: NotificationService) {
    super(_store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.preservationSpace_notification, preservationSpaceNotificationActionNameSpace, {
      canCreate: false,
      canGoBack: false,
      canRefresh: true,
    }, LocalStateEnum.preservationSpace);
  }

  ngOnInit(): void {
    this._retrieveModeFromUrl();
    super.ngOnInit();
  }

  private _retrieveModeFromUrl(): void {
    let orgUnitId = undefined;
    const queryParam = this._route.snapshot.queryParamMap;
    if (queryParam.has(NotificationHelper.KEY_ORGUNIT_ID)) {
      this.columnsSkippedToClear = [this.KEY_QUERY_PARAMETER_NOTIFIED_ORGUNIT_ID];
      orgUnitId = queryParam.get(NotificationHelper.KEY_ORGUNIT_ID);
    }
    this.mode = NotificationHelper.getMode(this._route);
    this._store.dispatch(new PreservationSpaceNotificationAction.SetMode(this.mode, orgUnitId));
  }

  @Override()
  showDetail(model: NotificationDlcm): void {
    //if it is a notification request, it is redirected directly to the object. If not to the object detail
    if (model.notificationType.notificationCategory === Enums.Notification.CategoryEnum.REQUEST) {
      this._goToDetailObject(model);
    } else {
      this._store.dispatch(new PreservationSpaceNotificationAction.SetProcessed(model.resId, model.notificationType.notificationCategory, ViewModeEnum.list));
      NotificationHelper.process(model, this._store, this._notificationService);
    }
  }

  conditionDisplayEditButton(model: NotificationDlcm | undefined): boolean {
    return false;
  }

  conditionDisplayDeleteButton(model: NotificationDlcm | undefined): boolean {
    return false;
  }

  setUnread(notification: NotificationDlcm): void {
    this._store.dispatch(new PreservationSpaceNotificationAction.SetPending(notification.resId, notification.notificationType.notificationCategory, ViewModeEnum.list));
  }

  private defineActionsInbox(): DataTableActions<NotificationDlcm>[] {
    return [
      {
        logo: IconNameEnum.markAsUnread,
        callback: model => this.setUnread(model),
        placeholder: current => isNotNullOrUndefined(current) && MARK_AS_TRANSLATABLE("preservationSpace.notifications.navigation.button.markAsUnread"),
        displayOnCondition: model => this._isRootOrAdminOrManagerOrSteward(model?.notifiedOrgUnit?.resId) && model.notificationStatus !== Enums.Notification.StatusEnum.PENDING,
        isWrapped: true,
      },
    ];
  }

  private _isRootOrAdminOrManagerOrSteward(notificationId: string): boolean {
    return isNotNullOrUndefined(notificationId) && this._securityService.isRootOrAdmin() || this._securityService.isManagerOfOrgUnit(notificationId) || this._securityService.isStewardOfOrgUnit(notificationId);
  }

  private _goToDetailObject(model: NotificationDlcm): void {
    const url = this._route.snapshot.parent.url.toString();
    let route = RoutesEnum.preservationSpaceNotificationInboxDetail;
    if (url === PreservationSpaceRoutesEnum.notificationSent) {
      route = RoutesEnum.preservationSpaceNotificationSentDetail;
    }
    this._store.dispatch(new Navigate([route, model.resId]));
  }

  protected defineActions(): DataTableActions<NotificationDlcm>[] {
    if (this.mode === NotificationModeEnum.inbox) {
      return this.defineActionsInbox();
    } else {
      return [];
    }
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "notificationType.notificationCategory" as any,
        header: LabelTranslateEnum.category,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.Notification.CategoryEnumTranslate,
      },
      {
        field: "notificationType.resId" as any,
        header: LabelTranslateEnum.type,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.Notification.TypeEnumTranslate,
      },
      {
        field: "emitter.person.fullName" as any,
        header: LabelTranslateEnum.emitter,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: false,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: false,
        isSortable: true,
      },
    ];

    const orgUnitId = MemoizedUtil.selectSnapshot(this._store, PreservationSpaceNotificationState, state => state.orgUnitId);
    if (isNullOrUndefined(orgUnitId)) {
      this.columns.push({
        field: "notifiedOrgUnit.name" as any,
        header: LabelTranslateEnum.notifiedOrganizationalUnit,
        type: FieldTypeEnum.searchableSingleSelect,
        order: OrderEnum.none,
        filterableField: this.KEY_QUERY_PARAMETER_NOTIFIED_ORGUNIT_ID as any,
        sortableField: this.KEY_QUERY_PARAMETER_NOTIFIED_ORGUNIT_ID as any,
        resourceNameSpace: this.sharedOrgUnitActionNameSpace,
        resourceState: this.sharedOrganizationalUnitState as any,
        searchableSingleSelectSort: this.sharedOrgUnitSort,
        isFilterable: true,
        isSortable: true,
      });
    }
  }
}
