import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {NotificationModeEnum} from "@app/features/preservation-space/notification/enums/notification-mode.enum";
import {NotificationHelper} from "@app/features/preservation-space/notification/helper/notification.helper";
import {
  PreservationSpaceNotificationState,
  PreservationSpaceNotificationStateModel,
} from "@app/features/preservation-space/notification/stores/preservation-space-notification.state";
import {PreservationSpaceNotificationStatusHistoryAction} from "@app/features/preservation-space/notification/stores/status-history/preservation-space-notification-status-history.action";
import {PreservationSpaceNotificationStatusHistoryState} from "@app/features/preservation-space/notification/stores/status-history/preservation-space-notification-status-history.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  NotificationDlcm,
  Role,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditCommonRoutable} from "@shared/components/routables/shared-abstract-detail-edit-common/shared-abstract-detail-edit-common.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  PreservationSpaceRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {ViewModeEnum} from "@shared/enums/view-mode.enum";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {sharedNotificationActionNameSpace} from "@shared/stores/notification/shared-notification.action";
import {SharedRoleState} from "@shared/stores/role/shared-role.state";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {Observable} from "rxjs";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  Override,
  OverrideProperty,
  ResourceNameSpace,
} from "solidify-frontend";
import {QueryParameters} from "solidify-frontend/lib/models";
import {
  PreservationSpaceNotificationAction,
  preservationSpaceNotificationActionNameSpace,
} from "../../../stores/preservation-space-notification.action";

@Component({
  selector: "dlcm-preservation-space-notification-detail-edit-routable",
  templateUrl: "./preservation-space-notification-detail-edit.routable.html",
  styleUrls: ["../../../../../../shared/components/routables/shared-abstract-detail-edit-common/shared-abstract-detail-edit-common.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationSpaceNotificationDetailEditRoutable extends SharedAbstractDetailEditCommonRoutable<NotificationDlcm, PreservationSpaceNotificationStateModel> implements OnInit {
  @Select(PreservationSpaceNotificationState.isLoadingWithDependency) isLoadingWithDependencyObs: Observable<boolean>;
  @Select(PreservationSpaceNotificationState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;

  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, PreservationSpaceNotificationStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationSpaceNotificationStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationSpaceNotificationStatusHistoryState, state => state.queryParameters);
  listRoleObs: Observable<Role[]> = MemoizedUtil.list(this._store, SharedRoleState);

  @OverrideProperty()
  checkAvailableResourceNameSpace: ResourceNameSpace = sharedNotificationActionNameSpace;

  @OverrideProperty()
  readonly editAvailable: boolean = false;
  @OverrideProperty()
  readonly deleteAvailable: boolean = false;

  mode: NotificationModeEnum;
  modeObs: Observable<NotificationModeEnum> = MemoizedUtil.select(this._store, PreservationSpaceNotificationState, state => state.mode);
  readonly KEY_PARAM_NAME: keyof NotificationDlcm & string = "notificationType";

  listExtraButtons: ExtraButtonToolbar<NotificationDlcm>[] = [
    {
      color: "primary",
      icon: IconNameEnum.navigate,
      displayCondition: current => !isNullOrUndefined(current) && this.mode === NotificationModeEnum.inbox && (current.notificationStatus === Enums.Notification.StatusEnum.PENDING),
      callback: (current) => NotificationHelper.process(current, this._store, this._notificationService),
      labelToTranslate: (current) => isNotNullOrUndefined(current) && MARK_AS_TRANSLATABLE("preservationSpace.notifications.navigation.button.process"),
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.approve,
      displayCondition: current => !isNullOrUndefined(current) && this.mode === NotificationModeEnum.inbox && current.notificationStatus === Enums.Notification.StatusEnum.PENDING,
      callback: (current) => this._setProcessed(current),
      labelToTranslate: (current) => LabelTranslateEnum.markAsDone,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.unapprove,
      displayCondition: current => !isNullOrUndefined(current) && this.mode === NotificationModeEnum.inbox && current.notificationStatus === Enums.Notification.StatusEnum.PENDING,
      callback: (current) => this._setRefuse(current),
      labelToTranslate: (current) => LabelTranslateEnum.markAsRefused,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.markAsUnread,
      displayCondition: current => !isNullOrUndefined(current) && this.mode === NotificationModeEnum.inbox && current.notificationStatus !== Enums.Notification.StatusEnum.PENDING,
      callback: (current) => this._setPending(current),
      labelToTranslate: (current) => isNotNullOrUndefined(current) && MARK_AS_TRANSLATABLE("preservationSpace.notifications.navigation.button.markAsUnread"),
      order: 40,
    },
  ];

  constructor(protected _store: Store,
              protected _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              private readonly _notificationService: NotificationService,
              public _dialog: MatDialog) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.preservationSpace_notification, preservationSpaceNotificationActionNameSpace, LocalStateEnum.preservationSpace);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this._retrieveModeFromUrl();
    this.subscribe(this.modeObs, mode => this.mode = mode);
  }

  getSubResourceWithParentId(id: string): void {
  }

  private _retrieveModeFromUrl(): void {
    this._store.dispatch(new PreservationSpaceNotificationAction.SetMode(NotificationHelper.getMode(this._route), undefined));
  }

  @Override()
  backToList(): void {
    const url = this._route.snapshot.parent.url.toString();
    this.mode = NotificationModeEnum.inbox;
    let route = RoutesEnum.preservationSpaceNotificationInbox;
    if (url === PreservationSpaceRoutesEnum.notificationSent) {
      route = RoutesEnum.preservationSpaceNotificationSent;
      this.mode = NotificationModeEnum.sent;
    }
    const orgUnitId = MemoizedUtil.selectSnapshot(this._store, PreservationSpaceNotificationState, state => state.orgUnitId);
    this._store.dispatch(new Navigate([route], NotificationHelper.getUrlQueryParam(orgUnitId)));
  }

  private _setProcessed(notification: NotificationDlcm): void {
    this._store.dispatch(new PreservationSpaceNotificationAction.SetProcessed(notification.resId, notification.notificationType.notificationCategory, ViewModeEnum.detail));
  }

  private _setRefuse(notification: NotificationDlcm): void {
    this._store.dispatch(new PreservationSpaceNotificationAction.SetRefuse(notification.resId, notification.notificationType.notificationCategory, ViewModeEnum.detail));
  }

  private _setPending(notification: NotificationDlcm): void {
    this._store.dispatch(new PreservationSpaceNotificationAction.SetPending(notification.resId, notification.notificationType.notificationCategory, ViewModeEnum.detail));
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: PreservationSpaceNotificationStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

}
