import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {IngestResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {Sip} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {PreservationPlanningSipCollectionState} from "@preservation-planning/sip/stores/collection/preservation-planning-sip-collection.state";
import {
  defaultPreservationPlanningSipDataFileValue,
  PreservationPlanningSipDataFileState,
  PreservationPlanningSipDataFileStateModel,
} from "@preservation-planning/sip/stores/data-file/preservation-planning-sip-data-file.state";
import {
  PreservationPlanningSipAction,
  preservationPlanningSipActionNameSpace,
} from "@preservation-planning/sip/stores/preservation-planning-sip.action";
import {
  PreservationPlanningSipStatusHistoryState,
  PreservationPlanningSipStatusHistoryStateModel,
} from "@preservation-planning/sip/stores/status-history/preservation-planning-sip-status-history.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";

export interface PreservationPlanningSipStateModel extends ResourceStateModel<Sip> {
  [LocalStateEnum.preservationPlanning_sip_dataFile]: PreservationPlanningSipDataFileStateModel;
  [LocalStateEnum.preservationPlanning_sip_statusHistory]: PreservationPlanningSipStatusHistoryStateModel;
  isLoadingDataFile: boolean;
}

@Injectable()
@State<PreservationPlanningSipStateModel>({
  name: LocalStateEnum.preservationPlanning_sip,
  defaults: {
    ...defaultResourceStateInitValue(),
    [LocalStateEnum.preservationPlanning_sip_dataFile]: defaultPreservationPlanningSipDataFileValue(),
    [LocalStateEnum.preservationPlanning_sip_statusHistory]: {...defaultStatusHistoryInitValue()},
    isLoadingDataFile: false,
  },
  children: [
    PreservationPlanningSipDataFileState,
    PreservationPlanningSipStatusHistoryState,
    PreservationPlanningSipCollectionState,
  ],
})
export class PreservationPlanningSipState extends ResourceState<PreservationPlanningSipStateModel, Sip> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected httpClient: HttpClient,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningSipActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return IngestResourceApiEnum.sip;
  }

  @Selector()
  static isLoading(state: PreservationPlanningSipStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: PreservationPlanningSipStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.info.name;
  }

  @Selector()
  static isLoadingWithDependency(state: PreservationPlanningSipStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: PreservationPlanningSipStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: PreservationPlanningSipStateModel): boolean {
    return true;
  }

  @Action(PreservationPlanningSipAction.Download)
  download(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.Download): void {
    const fileName = "sip_" + action.id + ".zip";
    this.downloadService.download(false, `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`, fileName);
  }

  @Action(PreservationPlanningSipAction.Resume)
  resume(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.httpClient.post<Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.RESUME}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new PreservationPlanningSipAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new PreservationPlanningSipAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningSipAction.ResumeFail(action));
          throw error;
        }),
      );
  }

  @Action(PreservationPlanningSipAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.sip.action.resume.success"));
  }

  @Action(PreservationPlanningSipAction.ResumeFail)
  resumeFail(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.sip.action.resume.fail"));
  }

  @Action(PreservationPlanningSipAction.Resubmit)
  resubmit(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.Resubmit): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.httpClient.post<Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.RESUBMIT}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new PreservationPlanningSipAction.ResubmitSuccess(action));
          } else {
            ctx.dispatch(new PreservationPlanningSipAction.ResubmitFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningSipAction.ResubmitFail(action));
          throw error;
        }),
      );
  }

  @Action(PreservationPlanningSipAction.ResubmitSuccess)
  resubmitSuccess(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.ResubmitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.sip.action.resubmit.success"));
  }

  @Action(PreservationPlanningSipAction.ResubmitFail)
  resubmitFail(ctx: StateContext<PreservationPlanningSipStateModel>, action: PreservationPlanningSipAction.ResubmitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.sip.action.resubmit.fail"));
  }
}
