import {
  Inject,
  Injectable,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {ApiActionEnum} from "@app/shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@app/shared/enums/api-resource-name.enum";
import {IngestResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {SipDataFile} from "@preservation-planning/sip/models/sip-data-file.model";
import {
  PreservationPlanningSipDataFileAction,
  preservationPlanningSipDataFileActionNameSpace,
} from "@preservation-planning/sip/stores/data-file/preservation-planning-sip-data-file.action";
import {
  PreservationPlanningSipDataFileStatusHistoryState,
  SharedAipDataFileStatusHistoryStateModel,
} from "@preservation-planning/sip/stores/data-file/status-history/preservation-planning-sip-data-file-status-history.state";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  NotificationService,
  SolidifyStateError,
} from "solidify-frontend";

export const defaultPreservationPlanningSipDataFileValue: () => PreservationPlanningSipDataFileStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.preservationPlanning_sip_dataFile_statusHistory]: defaultStatusHistoryInitValue(),
  });

export interface PreservationPlanningSipDataFileStateModel extends CompositionStateModel<SipDataFile> {
  [LocalStateEnum.preservationPlanning_sip_dataFile_statusHistory]: SharedAipDataFileStatusHistoryStateModel;
}

@Injectable()
@State<PreservationPlanningSipDataFileStateModel>({
  name: LocalStateEnum.preservationPlanning_sip_dataFile,
  defaults: {
    ...defaultPreservationPlanningSipDataFileValue(),
  },
  children: [
    PreservationPlanningSipDataFileStatusHistoryState,
  ],
})
export class PreservationPlanningSipDataFileState extends CompositionState<PreservationPlanningSipDataFileStateModel, SipDataFile> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              @Inject(WINDOW) private _window: Window,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningSipDataFileActionNameSpace,
      resourceName: ApiResourceNameEnum.DATAFILE,
    });
  }

  protected get _urlResource(): string {
    return IngestResourceApiEnum.sip;
  }

  @Action(PreservationPlanningSipDataFileAction.Refresh)
  refresh(ctx: StateContext<PreservationPlanningSipDataFileStateModel>, action: PreservationPlanningSipDataFileAction.Refresh): void {
    ctx.dispatch(new PreservationPlanningSipDataFileAction.GetAll(action.parentId, undefined, true));
  }

  @Action(PreservationPlanningSipDataFileAction.Download)
  download(ctx: StateContext<PreservationPlanningSipDataFileStateModel>, action: PreservationPlanningSipDataFileAction.Download): void {
    const url = `${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.DL}`;
    this.downloadService.download(false, url, action.dataFile.fileName, action.dataFile.fileSize);
  }

  @Action(PreservationPlanningSipDataFileAction.Resume)
  resume(ctx: StateContext<PreservationPlanningSipDataFileStateModel>, action: PreservationPlanningSipDataFileAction.Resume): Observable<Result> {
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.RESUME}`)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            this.notificationService.showInformation(LabelTranslateEnum.resourceResumed);
            ctx.dispatch(new PreservationPlanningSipDataFileAction.GetAll(action.parentId, undefined, true));
          } else {
            this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          }
        }),
        catchError(error => {
          this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          throw new SolidifyStateError(this, error);
        }),
      );
  }
}
