import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {Sip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {PreservationPlanningSipFormPresentational} from "@preservation-planning/sip/components/presentationals/sip-form/preservation-planning-sip-form.presentational";
import {SipDataFile} from "@preservation-planning/sip/models/sip-data-file.model";
import {PreservationPlanningSipDataFileAction} from "@preservation-planning/sip/stores/data-file/preservation-planning-sip-data-file.action";
import {PreservationPlanningSipDataFileState} from "@preservation-planning/sip/stores/data-file/preservation-planning-sip-data-file.state";
import {preservationPlanningSipActionNameSpace} from "@preservation-planning/sip/stores/preservation-planning-sip.action";
import {PreservationPlanningSipStateModel} from "@preservation-planning/sip/stores/preservation-planning-sip.state";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  isNullOrUndefined,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-sip-file-routable",
  templateUrl: "./preservation-planning-sip-file.routable.html",
  styleUrls: ["./preservation-planning-sip-file.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationPlanningSipFileRoutable extends SharedAbstractDetailEditRoutable<Sip, PreservationPlanningSipStateModel> implements OnInit {
  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningSipDataFileState);
  listDataFileObs: Observable<SipDataFile[]> = MemoizedUtil.list(this._store, PreservationPlanningSipDataFileState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, PreservationPlanningSipDataFileState);

  isInDetailMode: boolean = false;

  @ViewChild("formPresentational")
  readonly formPresentational: PreservationPlanningSipFormPresentational;

  columns: DataTableColumns<SipDataFile>[];
  actions: DataTableActions<SipDataFile>[];

  readonly KEY_PARAM_NAME: keyof Sip & string = undefined;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.preservationPlanning_sip, preservationPlanningSipActionNameSpace, LocalStateEnum.preservationPlanning);

    this.columns = [
      {
        field: "fileName",
        header: LabelTranslateEnum.fileName,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.DataFile.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
      {
        field: "complianceLevel",
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
    ];

    this.actions = [
      {
        logo: IconNameEnum.resume,
        callback: (sipDataFile: SipDataFile) => this.resumeDataFile(this._resId, sipDataFile),
        placeholder: current => LabelTranslateEnum.resume,
        displayOnCondition: current => isNullOrUndefined(current) || current.status === Enums.DataFile.StatusEnum.IN_ERROR,
      },
      {
        logo: IconNameEnum.download,
        callback: (sipDataFile: SipDataFile) => this.downloadDataFile(this._resId, sipDataFile),
        placeholder: current => LabelTranslateEnum.download,
        displayOnCondition: (sipDataFile: SipDataFile) => true,
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();

    this.subscribe(this._store.select((s: LocalStateModel) => s.router.state.url)
      .pipe(
        distinctUntilChanged(),
        tap(url => {
          this.isInDetailMode = !url.endsWith(PreservationPlanningRoutesEnum.sipFiles);
          this._changeDetector.detectChanges();
        }),
      ),
    );
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getDepositById(this._resId);
  }

  private getDepositById(id: string): void {
    // this._store.dispatch(ResourceActionHelper.getById(preservationSipActionNameSpace, id, true));
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    // const queryParameter = new QueryParameters();
    // MappingObjectUtil.set(queryParameter.search.searchItems, "relativeLocation", "/");
    this._store.dispatch(new PreservationPlanningSipDataFileAction.GetAll(id, null, true));
    // this._store.dispatch(new PreservationSipDataFileAction.GetListFolder(id));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new PreservationPlanningSipDataFileAction.ChangeQueryParameters(this._resId, queryParameters, true));
    // this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  refresh(): void {
    this._store.dispatch(new PreservationPlanningSipDataFileAction.Refresh(this._resId));
  }

  download($event: SipDataFile): void {
    this._store.dispatch(new PreservationPlanningSipDataFileAction.Download(this._resId, $event));
  }

  showDetail(sipDataFile: SipDataFile): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningSipDetail, this._resId, PreservationPlanningRoutesEnum.sipFiles, sipDataFile.resId]));
  }

  private downloadDataFile(parentId: string, dataFile: SipDataFile): void {
    this._store.dispatch(new PreservationPlanningSipDataFileAction.Download(parentId, dataFile));
  }

  private resumeDataFile(parentId: string, dataFile: SipDataFile): void {
    this._store.dispatch(new PreservationPlanningSipDataFileAction.Resume(parentId, dataFile));
  }
}
