import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {MatTabGroup} from "@angular/material/tabs";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Sip} from "@models";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {PreservationPlanningSipFormPresentational} from "@preservation-planning/sip/components/presentationals/sip-form/preservation-planning-sip-form.presentational";
import {
  PreservationPlanningSipAction,
  preservationPlanningSipActionNameSpace,
} from "@preservation-planning/sip/stores/preservation-planning-sip.action";
import {PreservationPlanningSipStateModel} from "@preservation-planning/sip/stores/preservation-planning-sip.state";
import {PreservationPlanningSipStatusHistoryAction} from "@preservation-planning/sip/stores/status-history/preservation-planning-sip-status-history.action";
import {PreservationPlanningSipStatusHistoryState} from "@preservation-planning/sip/stores/status-history/preservation-planning-sip-status-history.state";
import {Tab} from "@shared/components/containers/shared-tabs/shared-tabs.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {map} from "rxjs/operators";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OverrideProperty,
  QueryParameters,
} from "solidify-frontend";
import {LocalStateModel} from "@shared/models/local-state.model";
import {PermissionUtil} from "@shared/utils/permission.util";

@Component({
  selector: "dlcm-preservation-planning-sip-detail-edit-routable",
  templateUrl: "./preservation-planning-sip-detail-edit.routable.html",
  styleUrls: ["./preservation-planning-sip-detail-edit.routable.scss"],
})
export class PreservationPlanningSipDetailEditRoutable extends SharedAbstractDetailEditRoutable<Sip, PreservationPlanningSipStateModel> implements OnInit, OnDestroy {
  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, PreservationPlanningSipStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningSipStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationPlanningSipStatusHistoryState, state => state.queryParameters);

  userRolesObs: Enums.UserApplicationRole.UserApplicationRoleEnum[];

  @ViewChild("formPresentational")
  readonly formPresentational: PreservationPlanningSipFormPresentational;

  @ViewChild("matTabGroup")
  readonly matTabGroup: MatTabGroup;

  message: string;

  @OverrideProperty()
  readonly editAvailable: boolean = false;

  @OverrideProperty()
  readonly deleteAvailable: boolean = false;

  readonly KEY_PARAM_NAME: keyof Sip & string = undefined;
  readonly KEY_DELETE_BUTTON: string = undefined;
  readonly KEY_EDIT_BUTTON: string = undefined;
  readonly KEY_BACK_BUTTON: string = MARK_AS_TRANSLATABLE("preservation.sip.button.goBackToList");
  readonly KEY_BACK_TO_DETAIL_BUTTON: string = undefined;

  private currentTab: Tab;

  private get rootUrl(): string[] {
    return [RoutesEnum.preservationPlanningSipDetail, this._resId];
  }

  listExtraButtons: ExtraButtonToolbar<Sip>[] = [
    {
      color: "primary",
      icon: IconNameEnum.download,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.COMPLETED,
      callback: () => this.download(),
      labelToTranslate: (current) => LabelTranslateEnum.download,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.resume,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.IN_ERROR,
      callback: () => this.resume(),
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.submit,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && this._securityService.isRoot(),
      callback: () => this.resubmit(),
      labelToTranslate: (current) => LabelTranslateEnum.resubmit,
      order: 40,
    },
  ];

  listTabs: Tab[] = [
    {
      id: TabEnum.METADATA,
      suffixUrl: PreservationPlanningRoutesEnum.sipMetadata,
      icon: IconNameEnum.metadata,
      titleToTranslate: LabelTranslateEnum.metadata,
      route: () => [...this.rootUrl, PreservationPlanningRoutesEnum.sipMetadata],
    },
    {
      id: TabEnum.FILES,
      suffixUrl: PreservationPlanningRoutesEnum.sipFiles,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.files,
      route: () => [...this.rootUrl, PreservationPlanningRoutesEnum.sipFiles],
      conditionDisplay: () => this.currentObs.pipe(map(current => current?.dataFileNumber > 0)),
    },
    {
      id: TabEnum.COLLECTION,
      suffixUrl: PreservationPlanningRoutesEnum.sipCollections,
      icon: IconNameEnum.collection,
      titleToTranslate: LabelTranslateEnum.collection,
      route: () => [...this.rootUrl, PreservationPlanningRoutesEnum.sipCollections],
      conditionDisplay: () => this.currentObs.pipe(map(current => current?.collectionSize > 0)),
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router,
              private readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.preservationPlanning_sip, preservationPlanningSipActionNameSpace, LocalStateEnum.preservationPlanning);
    this.userRolesObs = this._store.selectSnapshot((state: LocalStateModel) => state.application.userRoles);
  }

  ngOnInit(): void {
    super.ngOnInit();

    this.retrieveCurrentModelWithUrl();
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
  }

  getSubResourceWithParentId(id: string): void {
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: PreservationPlanningSipStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

  setCurrentTab($event: Tab): void {
    this.currentTab = $event;
  }

  download(): void {
    this._store.dispatch(new PreservationPlanningSipAction.Download(this._resId));
  }

  resume(): void {
    this._store.dispatch(new PreservationPlanningSipAction.Resume(this._resId));
  }

  resubmit(): void {
    this._store.dispatch(new PreservationPlanningSipAction.Resubmit(this._resId));
  }
}

enum TabEnum {
  METADATA = "METADATA",
  FILES = "FILES",
  COLLECTION = "COLLECTIONS",
}
