import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {
  Aip,
  Sip,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {PreservationPlanningSipFormPresentational} from "@preservation-planning/sip/components/presentationals/sip-form/preservation-planning-sip-form.presentational";
import {PreservationPlanningSipCollectionAction} from "@preservation-planning/sip/stores/collection/preservation-planning-sip-collection.action";
import {PreservationPlanningSipCollectionState} from "@preservation-planning/sip/stores/collection/preservation-planning-sip-collection.state";
import {preservationPlanningSipActionNameSpace} from "@preservation-planning/sip/stores/preservation-planning-sip.action";
import {PreservationPlanningSipStateModel} from "@preservation-planning/sip/stores/preservation-planning-sip.state";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {SharedAipCollectionAction} from "@shared/features/aip/stores/aip-collection/shared-aip-collection.action";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-sip-collection-routable",
  templateUrl: "./preservation-planning-sip-collection.routable.html",
  styleUrls: ["./preservation-planning-sip-collection.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationPlanningSipCollectionRoutable extends SharedAbstractDetailEditRoutable<Sip, PreservationPlanningSipStateModel> implements OnInit {
  isLoadingCollectionObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningSipCollectionState);
  listCollectionObs: Observable<Aip[]> = MemoizedUtil.list(this._store, PreservationPlanningSipCollectionState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, PreservationPlanningSipCollectionState);

  isInDetailMode: boolean = false;

  @ViewChild("formPresentational")
  readonly formPresentational: PreservationPlanningSipFormPresentational;

  columns: DataTableColumns<Aip>[];
  actions: DataTableActions<Aip>[];

  readonly KEY_PARAM_NAME: keyof Sip & string = undefined;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.preservationPlanning_sip, preservationPlanningSipActionNameSpace, LocalStateEnum.preservationPlanning);

    this.columns = [
      {
        field: "info.name" as any,
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "info.status" as any,
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.DataFile.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
      {
        field: "info.complianceLevel" as any,
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
    ];

    this.actions = [
      {
        logo: IconNameEnum.download,
        callback: (aip: Aip) => this.downloadSipAip(this._resId, aip),
        placeholder: current => LabelTranslateEnum.download,
        displayOnCondition: current => false,
      },
      {
        logo: IconNameEnum.internalLink,
        callback: (aip: Aip) => this.goToAip(aip),
        placeholder: current => LabelTranslateEnum.goToAip,
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();

    this.subscribe(this._store.select((s: LocalStateModel) => s.router.state.url)
      .pipe(
        distinctUntilChanged(),
        tap(url => {
          this.isInDetailMode = !url.endsWith(PreservationPlanningRoutesEnum.sipCollections);
          this._changeDetector.detectChanges();
        }),
      ),
    );
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getDepositById(this._resId);
  }

  private getDepositById(id: string): void {
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new PreservationPlanningSipCollectionAction.GetAll(id, null, true));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new PreservationPlanningSipCollectionAction.ChangeQueryParameters(this._resId, queryParameters, true));
  }

  refresh(): void {
    this._store.dispatch(new PreservationPlanningSipCollectionAction.Refresh(this._resId));
  }

  showDetail(aip: Aip): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningSipDetail, this._resId, PreservationPlanningRoutesEnum.sipCollections, aip.resId]));
  }

  private downloadSipAip(parentId: string, aip: Aip): void {
    this._store.dispatch(new SharedAipCollectionAction.DownloadAip(parentId, aip));
  }

  private goToAip(aip: Aip): void {
    this._store.dispatch(new PreservationPlanningSipCollectionAction.GoToAip(aip));
  }
}
