import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {SharedAbstractFormPresentational} from "@app/shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@app/shared/models/base-form-definition.model";
import {Enums} from "@enums";
import {
  License,
  Sip,
} from "@models";
import {BaseResourceApiEnum} from "@shared/enums/api.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {KeyValueInfo} from "@shared/models/key-value-info.model";
import {Storage} from "@shared/models/storage.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {sharedLicenseActionNameSpace} from "@shared/stores/license/shared-license.action";
import {SharedLicenseState} from "@shared/stores/license/shared-license.state";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  DateUtil,
  EnumUtil,
  isNullOrUndefined,
  KeyValue,
  ObservableUtil,
  PropertyName,
  ResourceNameSpace,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-sip-form",
  templateUrl: "./preservation-planning-sip-form.presentational.html",
  styleUrls: ["./preservation-planning-sip-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationPlanningSipFormPresentational extends SharedAbstractFormPresentational<Sip> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  accessEnumValues: KeyValue[] = Enums.Deposit.AccessEnumTranslate;
  dataSensitivityEnumValues: KeyValue[] = Enums.Deposit.DataSensitivityEnumTranslate;
  packageStatusEnumValuesTranslate: KeyValue[] = Enums.Package.StatusEnumTranslate;

  sharedLicenseActionNameSpace: ResourceNameSpace = sharedLicenseActionNameSpace;
  sharedLicenseState: typeof SharedLicenseState = SharedLicenseState;

  @Input()
  isReady: boolean = false;

  listStoragions: Storage[] = BaseResourceApiEnum.archivalStorageList;

  get packageStatusEnum(): typeof Enums.Package.StatusEnum {
    return Enums.Package.StatusEnum;
  }

  private readonly _orgUnitValueBS: BehaviorSubject<string | undefined> = new BehaviorSubject<string | undefined>(undefined);
  @Output("orgUnitChange")
  readonly orgUnitValueObs: Observable<string | undefined> = ObservableUtil.asObservable(this._orgUnitValueBS);

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  protected initNewForm(): void {
  }

  protected bindFormTo(sip: Sip): void {
    this.form = this._fb.group({
      [this.formDefinition.organizationalUnitName]: [sip.organizationalUnit.name, []],
      [this.formDefinition.resId]: [sip.resId, []],
      [this.formDefinition.name]: [sip.info.name, []],
      [this.formDefinition.year]: ["", []],
      [this.formDefinition.status]: [sip.info.status, []],
      [this.formDefinition.access]: [sip.info.access, []],
      [this.formDefinition.dataSensitivity]: [sip.info.dataSensitivity, []],
      [this.formDefinition.licenseId]: [sip.info.licenseId, []],
      [this.formDefinition.complianceLevel]: [sip.info.complianceLevel, []],
      [this.formDefinition.embargoAccess]: [sip.info.embargo?.access, []],
      [this.formDefinition.embargoNumberMonths]: [sip.info.embargo?.months, []],
      [this.formDefinition.embargoStartDate]: [isNullOrUndefined(sip.info.embargo?.startDate) ? "" : DateUtil.convertDateToDateTimeString(new Date(sip.info.embargo?.startDate))],
    });
    this.isValidWhenDisable = this.form.valid;
  }

  protected treatmentBeforeSubmit(sip: Sip): Sip {
    return sip;
  }

  getSensitivityTooltip(dataSensitivity: Enums.Deposit.DataSensitivityEnum): string {
    return (EnumUtil.getKeyValue(Enums.Deposit.DataSensitivityEnumTranslate, dataSensitivity) as KeyValueInfo)?.infoToTranslate;
  }

  goToDeposit(): void {
    this._navigateBS.next([RoutesEnum.deposit, this.model.organizationalUnitId, DepositRoutesEnum.detail, this.model.depositId]);
  }

  goToAip(storagion: Storage): void {
    this._navigateBS.next([AppRoutesEnum.preservationPlanning, PreservationPlanningRoutesEnum.aip, String(storagion.index), SharedAipRoutesEnum.aipDetail, this.model.aipId]);
  }

  licenceCallback: (value: License) => string = (value: License) => value.title;
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() organizationalUnitName: string;
  @PropertyName() resId: string;
  @PropertyName() name: string;
  @PropertyName() year: string;
  @PropertyName() status: string;
  @PropertyName() access: string;
  @PropertyName() dataSensitivity: string;
  @PropertyName() licenseId: string;
  @PropertyName() complianceLevel: string;
  @PropertyName() deposit: string;
  @PropertyName() aip: string;
  @PropertyName() embargoAccess: string;
  @PropertyName() embargoNumberMonths: string;
  @PropertyName() embargoStartDate: string;
}
