import {Injectable} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  preservationPlanningMonitorActionNameSpace,
  PreservationPlanningMonitoringAction,
} from "@preservation-planning/monitoring/stores/preservation-planning-monitoring.action";
import {
  PreservationPlanningMonitoringWebServiceState,
  PreservationPlanningMonitoringWebServiceStateModel,
} from "@preservation-planning/monitoring/stores/web-service/preservation-planning-monitoring-web-service.state";
import {PreservationPlanningResourceApiEnum} from "@shared/enums/api.enum";
import {PreservationPlanningMonitor} from "@shared/models/business/preservation-planning-monitor.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseState,
  BaseStateModel,
  defaultBaseStateInitValue,
  NotificationService,
  SolidifyStateError,
  StoreUtil,
} from "solidify-frontend";

export interface PreservationPlanningMonitoringStateModel extends BaseStateModel {
  preservationPlanningMonitor: PreservationPlanningMonitor | undefined;
  [LocalStateEnum.preservationPlanning_monitoring_webService]: PreservationPlanningMonitoringWebServiceStateModel | undefined;
}

@Injectable()
@State<PreservationPlanningMonitoringStateModel>({
  name: LocalStateEnum.preservationPlanning_monitoring,
  defaults: {
    ...defaultBaseStateInitValue(),
    preservationPlanningMonitor: undefined,
    [LocalStateEnum.preservationPlanning_monitoring_webService]: undefined,
  },
  children: [
    PreservationPlanningMonitoringWebServiceState,
  ],
})
export class PreservationPlanningMonitoringState extends BaseState<PreservationPlanningMonitoringStateModel> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningMonitorActionNameSpace,
    }, PreservationPlanningMonitoringState);
  }

  protected get _urlResource(): string {
    return PreservationPlanningResourceApiEnum.monitor;
  }

  @Action(PreservationPlanningMonitoringAction.Get)
  get(ctx: StateContext<PreservationPlanningMonitoringStateModel>, action: PreservationPlanningMonitoringAction.Get): Observable<PreservationPlanningMonitor> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        preservationPlanningMonitor: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    return this.apiService.get<PreservationPlanningMonitor>(this._urlResource, null)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [Navigate]),
        tap((preservationPlanningMonitor: PreservationPlanningMonitor) => {
          ctx.dispatch(new PreservationPlanningMonitoringAction.GetSuccess(action, preservationPlanningMonitor));
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningMonitoringAction.GetFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationPlanningMonitoringAction.GetSuccess)
  getAllSuccess(ctx: StateContext<PreservationPlanningMonitoringStateModel>, action: PreservationPlanningMonitoringAction.GetSuccess): void {
    ctx.patchState({
      preservationPlanningMonitor: action.preservationPlanningMonitor,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(PreservationPlanningMonitoringAction.GetFail)
  getAllFail(ctx: StateContext<PreservationPlanningMonitoringStateModel>, action: PreservationPlanningMonitoringAction.GetFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
