import {Injectable} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  Collection,
  PreservationJob,
} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  defaultPreservationPlanningJobExecutionInitValue,
  PreservationPlanningJobExecutionState,
  PreservationPlanningJobExecutionStateModel,
} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.state";
import {
  PreservationPlanningJobAction,
  preservationPlanningJobActionNameSpace,
} from "@preservation-planning/job/stores/preservation-planning-job.action";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {PreservationPlanningResourceApiEnum} from "@shared/enums/api.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {JobExecution} from "@shared/models/business/job-execution.model";
import {JobRecurrence} from "@shared/models/business/job-recurrence.model";
import {JobType} from "@shared/models/business/job-type.model";
import {Result} from "@shared/models/business/result.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isEmptyArray,
  isNullOrUndefined,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  QueryParameters,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface PreservationPlanningJobStateModel extends ResourceStateModel<PreservationJob> {
  listJobTypes: JobType[] | undefined;
  listJobRecurrences: JobRecurrence[] | undefined;
  [LocalStateEnum.preservationPlanning_job_execution]: PreservationPlanningJobExecutionStateModel | undefined;
}

@Injectable()
@State<PreservationPlanningJobStateModel>({
  name: LocalStateEnum.preservationPlanning_job,
  defaults: {
    ...defaultResourceStateInitValue(),
    listJobTypes: undefined,
    listJobRecurrences: undefined,
    [LocalStateEnum.preservationPlanning_job_execution]: {...defaultPreservationPlanningJobExecutionInitValue()},
  },
  children: [
    PreservationPlanningJobExecutionState,
  ],
})
export class PreservationPlanningJobState extends ResourceState<PreservationPlanningJobStateModel, PreservationJob> {

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningJobActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.preservationPlanningJobDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.preservationPlanningJobDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.preservationPlanningJob,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservation.job.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservation.job.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("preservation.job.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return PreservationPlanningResourceApiEnum.preservationJobs;
  }

  @Selector()
  static isLoading(state: PreservationPlanningJobStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: PreservationPlanningJobStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: PreservationPlanningJobStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isReadyToBeDisplayed(state: PreservationPlanningJobStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: PreservationPlanningJobStateModel): boolean {
    return true;
  }

  @Action(PreservationPlanningJobAction.Init)
  init(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.Init): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + ApiActionEnum.INIT).pipe(
      tap(result => {
        if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
          ctx.dispatch(new PreservationPlanningJobAction.InitSuccess(action));
        } else {
          ctx.dispatch(new PreservationPlanningJobAction.InitFail(action));
        }
      }),
      catchError(error => {
        ctx.dispatch(new PreservationPlanningJobAction.InitFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationPlanningJobAction.InitSuccess)
  initSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.InitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservation.job.notification.init.success"));
  }

  @Action(PreservationPlanningJobAction.InitFail)
  initFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.InitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservation.job.notification.init.fail"));
  }

  @Action(PreservationPlanningJobAction.VerifyBeforeStart)
  verifyBeforeStart(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.VerifyBeforeStart): Observable<Collection> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const JOB_EXEC_STATUS: keyof JobExecution = "status";
    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    MappingObjectUtil.set(queryParameters.search.searchItems, JOB_EXEC_STATUS, Enums.PreservationJob.ExecutionStatusEnum.IN_PROGRESS);

    return this.apiService.get<JobExecution>(this._urlResource + urlSeparator + action.resId + urlSeparator + ApiResourceNameEnum.PRES_JOB_EXECUTION, queryParameters)
      .pipe(
        tap(result => {
          if (isEmptyArray(result._data)) {
            ctx.dispatch(new PreservationPlanningJobAction.VerifyBeforeStartSuccess(action));
          } else {
            ctx.dispatch(new PreservationPlanningJobAction.VerifyBeforeStartFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningJobAction.StartFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationPlanningJobAction.VerifyBeforeStartSuccess)
  verifyBeforeStartSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.VerifyBeforeStartSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new PreservationPlanningJobAction.Start((action.parentAction as PreservationPlanningJobAction.VerifyBeforeStart).resId));
  }

  @Action(PreservationPlanningJobAction.VerifyBeforeStartFail)
  verifyBeforeStartFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.VerifyBeforeStartFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservation.job.notification.verify.fail"));
  }

  @Action(PreservationPlanningJobAction.Start)
  start(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.Start): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + action.resId + urlSeparator + ApiActionEnum.START).pipe(
      tap(result => {
        if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
          ctx.dispatch(new PreservationPlanningJobAction.StartSuccess(action));
        } else {
          ctx.dispatch(new PreservationPlanningJobAction.StartFail(action));
        }
      }),
      catchError(error => {
        ctx.dispatch(new PreservationPlanningJobAction.StartFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationPlanningJobAction.StartSuccess)
  startSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.StartSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservation.job.notification.start.success"));
  }

  @Action(PreservationPlanningJobAction.StartFail)
  startFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.StartFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservation.job.notification.start.fail"));
  }

  @Action(PreservationPlanningJobAction.Resume)
  resume(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + action.resId + urlSeparator + ApiActionEnum.RESUME).pipe(
      tap(result => {
        if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
          ctx.dispatch(new PreservationPlanningJobAction.ResumeSuccess(action));
        } else {
          ctx.dispatch(new PreservationPlanningJobAction.ResumeFail(action));
        }
      }),
      catchError(error => {
        ctx.dispatch(new PreservationPlanningJobAction.ResumeFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationPlanningJobAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservation.job.notification.resume.success"));
  }

  @Action(PreservationPlanningJobAction.ResumeFail)
  resumeFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservation.job.notification.resume.fail"));
  }

  @Action(PreservationPlanningJobAction.ListJobTypes)
  listJobTypes(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobTypes): Observable<JobType[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.getList<JobType>(this._urlResource + urlSeparator + ApiActionEnum.LIST_JOB_TYPE, null).pipe(
      tap(result => {
        ctx.dispatch(new PreservationPlanningJobAction.ListJobTypesSuccess(action, result));
      }),
      catchError(error => {
        ctx.dispatch(new PreservationPlanningJobAction.ListJobTypesFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationPlanningJobAction.ListJobTypesSuccess)
  listJobTypesSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobTypesSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listJobTypes: action.listJobTypes,
    });
  }

  @Action(PreservationPlanningJobAction.ListJobTypesFail)
  listJobTypesFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobTypesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(PreservationPlanningJobAction.ListJobRecurrences)
  listJobRecurrences(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobRecurrences): Observable<JobRecurrence[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.getList<JobRecurrence>(this._urlResource + urlSeparator + ApiActionEnum.LIST_JOB_RECURRENCE, null).pipe(
      tap(result => {
        ctx.dispatch(new PreservationPlanningJobAction.ListJobRecurrencesSuccess(action, result));
      }),
      catchError(error => {
        ctx.dispatch(new PreservationPlanningJobAction.ListJobRecurrencesFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationPlanningJobAction.ListJobRecurrencesSuccess)
  listJobRecurrencesSuccess(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobRecurrencesSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listJobRecurrences: action.listJobRecurrences,
    });
  }

  @Action(PreservationPlanningJobAction.ListJobRecurrencesFail)
  listJobRecurrencesFail(ctx: StateContext<PreservationPlanningJobStateModel>, action: PreservationPlanningJobAction.ListJobRecurrencesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
