import {Injectable} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {PreservationPlanningJobExecutionReportAction} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.action";
import {
  defaultPreservationPlanningJobExecutionReportInitValue,
  PreservationPlanningJobExecutionReportState,
  PreservationPlanningJobExecutionReportStateModel,
} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.state";
import {
  preservationJobExecutionActionNameSpace,
  PreservationPlanningJobExecutionAction,
} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.action";
import {PreservationPlanningJobStateModel} from "@preservation-planning/job/stores/preservation-planning-job.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {PreservationPlanningResourceApiEnum} from "@shared/enums/api.enum";
import {JobExecutionReport} from "@shared/models/business/job-execution-report.model";
import {JobExecution} from "@shared/models/business/job-execution.model";
import {Result} from "@shared/models/business/result.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export const defaultPreservationPlanningJobExecutionInitValue: () => PreservationPlanningJobExecutionStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.preservationPlanning_job_execution_report]: {...defaultPreservationPlanningJobExecutionReportInitValue()},
  });

export interface PreservationPlanningJobExecutionStateModel extends CompositionStateModel<JobExecution> {
  [LocalStateEnum.preservationPlanning_job_execution_report]: PreservationPlanningJobExecutionReportStateModel;
}

@Injectable()
@State<PreservationPlanningJobExecutionStateModel>({
  name: LocalStateEnum.preservationPlanning_job_execution,
  defaults: {
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.preservationPlanning_job_execution_report]: {...defaultPreservationPlanningJobExecutionReportInitValue()},
  },
  children: [
    PreservationPlanningJobExecutionReportState,
  ],
})
export class PreservationPlanningJobExecutionState extends CompositionState<PreservationPlanningJobExecutionStateModel, JobExecution> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationJobExecutionActionNameSpace,
      resourceName: ApiResourceNameEnum.PRES_JOB_EXECUTION,
    });
  }

  @Selector()
  static currentTitle(state: PreservationPlanningJobExecutionStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return "Run " + state.current.runNumber;
  }

  @Selector()
  static isReadyToBeDisplayed(state: PreservationPlanningJobStateModel): boolean {
    return !isNullOrUndefined(state.current);
  }

  @Selector()
  static isLoading(state: PreservationPlanningJobStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static listReport(state: PreservationPlanningJobExecutionStateModel): JobExecutionReport[] {
    return state[LocalStateEnum.preservationPlanning_job_execution_report].reports;
  }

  protected get _urlResource(): string {
    return PreservationPlanningResourceApiEnum.preservationJobs;
  }

  @Action(PreservationPlanningJobExecutionAction.Resume)
  resume(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + action.parentId + urlSeparator + ApiResourceNameEnum.PRES_JOB_EXECUTION + urlSeparator + action.jobExecutionId + urlSeparator + ApiActionEnum.RESUME)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new PreservationPlanningJobExecutionAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new PreservationPlanningJobExecutionAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningJobExecutionAction.ResumeFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(PreservationPlanningJobExecutionAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("preservation.jobExecution.notification.resume.success"));
  }

  @Action(PreservationPlanningJobExecutionAction.ResumeFail)
  resumeFail(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("preservation.jobExecution.notification.resume.fail"));
  }

  @OverrideDefaultAction()
  @Action(PreservationPlanningJobExecutionAction.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionAction.GetByIdSuccess): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new PreservationPlanningJobExecutionReportAction.Report(action.model.resId, action.model.jobId));
  }
}
