import {Injectable} from "@angular/core";
import {environment} from "@environments/environment";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ReportNamespace} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report-namespace.model";
import {
  PreservationPlanningJobExecutionReportAction,
  preservationPlanningJobExecutionReportNamespace,
} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.action";
import {PreservationPlanningJobExecutionStateModel} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {PreservationPlanningResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {JobExecutionReportLine} from "@shared/models/business/job-execution-report-line.model";
import {JobExecutionReport} from "@shared/models/business/job-execution-report.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseState,
  BaseStateModel,
  CollectionTyped,
  defaultBaseStateInitValue,
  NotificationService,
  ObjectUtil,
  QueryParameters,
  RegisterDefaultAction,
  SolidifyStateError,
} from "solidify-frontend";

export const defaultPreservationPlanningJobExecutionReportInitValue: () => PreservationPlanningJobExecutionReportStateModel = () =>
  ({
    ...defaultBaseStateInitValue(),
    reports: [],
    queryParameters: new QueryParameters(environment.defaultPageSize),
    lineReport: [],
    queryParametersLines: new QueryParameters(environment.defaultPageSize),
  });

export interface PreservationPlanningJobExecutionReportStateModel extends BaseStateModel {
  reports: JobExecutionReport[];
  queryParameters: QueryParameters;
  lineReport: JobExecutionReportLine[];
  queryParametersLines: QueryParameters;
}

@Injectable()
@State<PreservationPlanningJobExecutionReportStateModel>({
  name: LocalStateEnum.preservationPlanning_job_execution_report,
  defaults: {
    ...defaultPreservationPlanningJobExecutionReportInitValue(),
  },
  children: [],
})
export class PreservationPlanningJobExecutionReportState extends BaseState<PreservationPlanningJobExecutionReportStateModel> {

  protected readonly _nameSpace: ReportNamespace;

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningJobExecutionReportNamespace,
    }, PreservationPlanningJobExecutionReportState);
  }

  protected get _urlResource(): string {
    return PreservationPlanningResourceApiEnum.preservationJobs;
  }

  getQueryParametersToApply(queryParameters: QueryParameters, queryParameterState: QueryParameters): QueryParameters {
    return queryParameters == null ? queryParameterState : queryParameters;
  }

  updateQueryParameters<T>(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, list: CollectionTyped<T> | null | undefined, queryParametersLine: boolean): QueryParameters {
    let queryParameters = null;
    if (queryParametersLine) {
      queryParameters = ObjectUtil.clone(ctx.getState().queryParametersLines);
    } else {
      queryParameters = ObjectUtil.clone(ctx.getState().queryParameters);
    }
    const paging = ObjectUtil.clone(queryParameters.paging);
    paging.length = list === null || list === undefined ? 0 : list._page.totalItems;
    queryParameters.paging = paging;
    return queryParameters;
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.Report)
  report(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.Report): Observable<CollectionTyped<JobExecutionReport>> {
    ctx.patchState({
      reports: [],
      lineReport: [],
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: this.getQueryParametersToApply(action.queryParameters, ctx.getState().queryParameters),
    });
    return this.apiService.get<JobExecutionReport>(this._urlResource + urlSeparator + action.parentId + urlSeparator + ApiResourceNameEnum.PRES_JOB_EXECUTION + urlSeparator + action.id + urlSeparator + ApiActionEnum.REPORT, ctx.getState().queryParameters)
      .pipe(
        tap((list: CollectionTyped<JobExecutionReport>) => {
          ctx.dispatch(new this._nameSpace.ReportSuccess(action, list));
        }),
        catchError(error => {
          ctx.dispatch(new this._nameSpace.ReportFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ReportSuccess)
  reportSuccess(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ReportSuccess): void {
    const queryParameters = this.updateQueryParameters(ctx, action.list, false);
    ctx.patchState({
      reports: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ReportFail)
  reportFail(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ReportFail): void {
    ctx.patchState({
      reports: [],
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ReportLines)
  reportLines(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ReportLines): Observable<CollectionTyped<JobExecutionReportLine>> {
    ctx.patchState({
      lineReport: [],
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParametersLines: this.getQueryParametersToApply(action.queryParameters, ctx.getState().queryParametersLines),
    });

    return this.apiService.get<JobExecutionReportLine>(this._urlResource + urlSeparator + action.parentId + urlSeparator + ApiResourceNameEnum.PRES_JOB_EXECUTION + urlSeparator + action.id + urlSeparator + ApiActionEnum.REPORT + urlSeparator + action.reportId + urlSeparator + "detail", action.queryParameters)
      .pipe(
        tap((list: CollectionTyped<JobExecutionReportLine>) => {
          ctx.dispatch(new this._nameSpace.ReportLinesSuccess(action, list));
        }),
        catchError(error => {
          ctx.dispatch(new this._nameSpace.ReportLinesFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ReportLinesSuccess)
  reportLinesSuccess(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ReportLinesSuccess): void {
    const queryParameters = this.updateQueryParameters(ctx, action.list, true);
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      lineReport: action.list._data,
      queryParametersLines: queryParameters,
    });
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ReportLinesFail)
  reportLinesFail(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ReportLinesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });

    ctx.dispatch(new this._nameSpace.Report(action.id, action.parentId, action.queryParameters));
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.ChangeQueryParametersLines)
  changeQueryParametersLines(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.ChangeQueryParametersLines): void {
    ctx.patchState({
      queryParametersLines: action.queryParameters,
    });

    ctx.dispatch(new this._nameSpace.ReportLines(action.id, action.parentId, action.reportId, action.queryParameters));
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.CleanQueryParameters)
  cleanQueryParameters(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.CleanQueryParameters): void {
    ctx.patchState({
      queryParameters: new QueryParameters(),
    });
  }

  @RegisterDefaultAction((reportNamespace: ReportNamespace) => reportNamespace.CleanQueryParametersLines)
  cleanQueryParametersLines(ctx: StateContext<PreservationPlanningJobExecutionReportStateModel>, action: PreservationPlanningJobExecutionReportAction.CleanQueryParametersLines): void {
    ctx.patchState({
      queryParametersLines: new QueryParameters(),
    });
  }

  @Action(PreservationPlanningJobExecutionReportAction.GoToAip)
  goToAip(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionReportAction.GoToAip): void {
    const pathAipDetail = RoutesEnum.preservationPlanningAip + urlSeparator + 1 + urlSeparator + SharedAipRoutesEnum.aipDetail + AppRoutesEnum.separator;
    const path = [pathAipDetail, action.aipId];
    ctx.dispatch(new Navigate(path));
  }

  @Action(PreservationPlanningJobExecutionReportAction.GoToSip)
  goToSip(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionReportAction.GoToSip): void {
    const pathSipDetail = RoutesEnum.preservationPlanningSip + urlSeparator + PreservationPlanningRoutesEnum.sipDetail + AppRoutesEnum.separator;
    const path = [pathSipDetail, action.sipId];
    ctx.dispatch(new Navigate(path));
  }

  @Action(PreservationPlanningJobExecutionReportAction.GoToDeposit)
  goToDeposit(ctx: StateContext<PreservationPlanningJobExecutionStateModel>, action: PreservationPlanningJobExecutionReportAction.GoToDeposit): void {
    const pathDepositDetail = RoutesEnum.preservationPlanningDeposit + urlSeparator + PreservationPlanningRoutesEnum.depositDetail + AppRoutesEnum.separator;
    const path = [pathDepositDetail, action.depositId];
    ctx.dispatch(new Navigate(path));
  }
}
