import {
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {PreservationJob} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {
  PreservationPlanningJobAction,
  preservationPlanningJobActionNameSpace,
} from "@preservation-planning/job/stores/preservation-planning-job.action";
import {PreservationPlanningJobStateModel} from "@preservation-planning/job/stores/preservation-planning-job.state";
import {SharedAbstractListRoutable} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {JobRecurrence} from "@shared/models/business/job-recurrence.model";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  OrderEnum,
  Override,
} from "solidify-frontend";
import JobRecurrenceEnum = JobRecurrence.JobRecurrenceEnum;

@Component({
  selector: "dlcm-preservation-planning-job-list-routable",
  templateUrl: "../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.html",
  styleUrls: ["../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.scss"],
})
export class PreservationPlanningJobListRoutable extends SharedAbstractListRoutable<PreservationJob, PreservationPlanningJobStateModel> implements OnInit {
  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  readonly KEY_BACK_BUTTON: string | undefined = LabelTranslateEnum.backToPreservationPlanning;
  readonly KEY_PARAM_NAME: keyof PreservationJob & string = "name";

  constructor(protected readonly _store: Store,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog) {
    super(_store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.preservationPlanning_job, preservationPlanningJobActionNameSpace, {
      listExtraButtons: [
        {
          color: "primary",
          icon: IconNameEnum.init,
          labelToTranslate: (current) => MARK_AS_TRANSLATABLE("preservation.job.button.init"),
          callback: () => this.init(),
          order: 40,
        },
      ],
    }, LocalStateEnum.preservationPlanning);
  }

  conditionDisplayEditButton(model: PreservationJob | undefined): boolean {
    return true;
  }

  conditionDisplayDeleteButton(model: PreservationJob | undefined): boolean {
    return true;
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.defineColumns();
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "name",
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "jobRecurrence" as any,
        header: LabelTranslateEnum.recurrence,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.PreservationJob.RecurrenceEnumTranslate,
      },
      {
        field: "jobType",
        header: LabelTranslateEnum.type,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.PreservationJob.TypeEnumTranslate,
      },
      {
        field: "enable" as any,
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.boolean,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "lastUpdate.when" as any,
        header: LabelTranslateEnum.updated,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: true,
        isSortable: true,
      },
    ];
  }

  @Override()
  protected defineActions(): DataTableActions<PreservationJob>[] {
    return [
      ...super.defineActions(),
      {
        logo: IconNameEnum.run,
        callback: (model: PreservationJob) => this.start(model),
        placeholder: current => LabelTranslateEnum.run,
        displayOnCondition: (model: PreservationJob) => isNullOrUndefined(model) || (model.jobRecurrence !== JobRecurrenceEnum.ONCE),
      },
    ];
  }

  back(): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationPlanning]));
  }

  init(): void {
    this._store.dispatch(new PreservationPlanningJobAction.Init());
  }

  private start(job: PreservationJob): void {
    this._store.dispatch(new PreservationPlanningJobAction.VerifyBeforeStart(job.resId));
  }
}
