import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Select,
  Store,
} from "@ngxs/store";
import {
  PreservationPlanningJobReportDetailDialog,
  ReportLinesWrapper,
} from "@preservation-planning/job/components/dialogs/job-report-detail/preservation-planning-job-report-detail.dialog";
import {PreservationPlanningJobExecutionReportAction} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.action";
import {PreservationPlanningJobExecutionReportState} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.state";
import {PreservationPlanningJobExecutionAction} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.action";
import {PreservationPlanningJobExecutionState} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.state";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {JobExecutionReport} from "@shared/models/business/job-execution-report.model";
import {JobExecution} from "@shared/models/business/job-execution.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResource,
  CompositionState,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";
import {DepositState} from "@deposit/stores/deposit.state";
import {PreservationPlanningJobAction} from "@preservation-planning/job/stores/preservation-planning-job.action";
import {PreservationPlanningJobState} from "@preservation-planning/job/stores/preservation-planning-job.state";

@Component({
  selector: "dlcm-preservation-planning-job-execution-detail-routable",
  templateUrl: "./preservation-planning-job-execution-detail.routable.html",
  styleUrls: ["./preservation-planning-job-execution-detail.routable.scss"],
})
export class PreservationPlanningJobExecutionDetailRoutable extends SharedAbstractRoutable implements OnInit, OnDestroy {
  currentObs: Observable<JobExecution> = MemoizedUtil.current(this._store, PreservationPlanningJobExecutionState);
  @Select(PreservationPlanningJobExecutionState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningJobExecutionState);
  isLoadingReportObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningJobExecutionReportState);
  listReportObs: Observable<JobExecutionReport[]> = MemoizedUtil.select(this._store, PreservationPlanningJobExecutionReportState, state => state.reports);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationPlanningJobExecutionReportState, state => state.queryParameters);

  readonly editAvailable: boolean = false;
  readonly deleteAvailable: boolean = false;

  columns: DataTableColumns<BaseResource & JobExecutionReport>[] = [
    {
      field: "executionNumber",
      header: MARK_AS_TRANSLATABLE("preservation.jobExecution.report.table.header.executionNumber"),
      type: FieldTypeEnum.string,
      order: OrderEnum.descending,
      isFilterable: false,
      isSortable: true,
    },
    {
      field: "processedItems",
      header: MARK_AS_TRANSLATABLE("preservation.jobExecution.report.table.header.processedItems"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "ignoredItems",
      header: MARK_AS_TRANSLATABLE("preservation.jobExecution.report.table.header.ignoredItems"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
    {
      field: "inErrorItems",
      header: MARK_AS_TRANSLATABLE("preservation.jobExecution.report.table.header.inErrorItems"),
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isFilterable: false,
      isSortable: false,
    },
  ];

  private _preservationJobId: string;
  private _resId: string;

  constructor(private readonly _store: Store,
              private readonly _route: ActivatedRoute,
              protected readonly _changeDetector: ChangeDetectorRef,
              private readonly _actions$: Actions,
              private readonly _dialog: MatDialog) {
    super();
    this.retrieveCurrentModelWithUrl();
  }

  protected retrieveCurrentModelWithUrl(): void {
    this.retrieveResIdFromUrl();
    this._store.dispatch(new PreservationPlanningJobExecutionAction.GetById(this._preservationJobId, this._resId));
  }

  protected retrieveResIdFromUrl(): void {
    this._resId = this._route.snapshot.paramMap.get(AppRoutesEnum.paramIdExecutionWithoutPrefixParam);
    this._preservationJobId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  ngOnDestroy(): void {
    this._store.dispatch(new PreservationPlanningJobExecutionReportAction.CleanQueryParameters());
  }

  backToList(): void {
    const pathExecution = RoutesEnum.preservationPlanning + AppRoutesEnum.separator + PreservationPlanningRoutesEnum.job + AppRoutesEnum.separator + PreservationPlanningRoutesEnum.jobDetail;
    const path = [pathExecution, this._preservationJobId];
    this._store.dispatch(new Navigate(path));
  }

  listExtraButtons(): ExtraButtonToolbar<JobExecution>[] {
    return [
      {
        color: "primary",
        icon: IconNameEnum.resume,
        labelToTranslate: (current) => LabelTranslateEnum.resume,
        callback: () => this.resume(),
        displayCondition: () => !isNullOrUndefined(CompositionState.currentSnapshot(this._store, PreservationPlanningJobExecutionState)) && CompositionState.currentSnapshot(this._store, PreservationPlanningJobExecutionState).status === Enums.PreservationJob.ExecutionStatusEnum.IN_ERROR,
        order: 40,
      },
    ];
  }

  private resume(): void {
    this._store.dispatch(new PreservationPlanningJobExecutionAction.Resume(this._preservationJobId, this._resId));
    this.subscribe(this._actions$.pipe(ofActionCompleted(PreservationPlanningJobExecutionAction.ResumeSuccess))
      .pipe(
        take(1),
        tap(result => {
          if (result.result.successful) {
            this.refresh();
          }
        }),
      ));

    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: 2,
      incrementInterval: true,
      resetIntervalWhenUserMouseEvent: true,
      continueUntil: () => isNullOrUndefined(CompositionState.currentSnapshot(this._store, PreservationPlanningJobExecutionState)) || (CompositionState.currentSnapshot(this._store, PreservationPlanningJobExecutionState).status !== Enums.PreservationJob.ExecutionStatusEnum.IN_ERROR &&
        CompositionState.currentSnapshot(this._store, PreservationPlanningJobExecutionState).status !== Enums.PreservationJob.ExecutionStatusEnum.COMPLETED),
      actionToDo: () => this.refresh(),
    }));
  }

  showReportDetail(report: JobExecutionReport): void {
    const jobType = MemoizedUtil.currentSnapshot(this._store, PreservationPlanningJobState).jobType;

    this.subscribe(this._store.dispatch(new PreservationPlanningJobExecutionReportAction.ReportLines(this._resId, this._preservationJobId, report.resId, null))
      .pipe(
        tap((state: LocalStateModel) => {
          const reportLines = state.preservationPlanning.preservationPlanning_job.preservationPlanning_job_execution.preservationPlanning_job_execution_report.lineReport;
          this._dialog.open(PreservationPlanningJobReportDetailDialog, {
            data: {
              linesReport: reportLines,
              preservationJobId: this._preservationJobId,
              executionId: this._resId,
              reportId: report.resId,
              jobType: jobType,
            } as ReportLinesWrapper,
          });
        }),
      ));
  }

  refresh(): void {
    this._store.dispatch(new PreservationPlanningJobExecutionAction.GetById(this._preservationJobId, this._resId));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new PreservationPlanningJobExecutionReportAction.ChangeQueryParameters(queryParameters, this._resId, this._preservationJobId));
    this._changeDetector.detectChanges();
  }
}
