import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  ViewChild,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {JobHelper} from "@app/features/preservation-planning/job/helper/job.helper";
import {Enums} from "@enums";
import {
  PreservationJob,
  PreservationJobScheduling,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {
  isNullOrUndefined,
  KeyValue,
  ObjectUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-job-form",
  templateUrl: "./preservation-planning-job-form.presentational.html",
  styleUrls: ["../../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationPlanningJobFormPresentational extends SharedAbstractFormPresentational<PreservationJob> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  listJobType: KeyValue[] = Enums.PreservationJob.TypeEnumTranslate;
  listJobRecurrence: KeyValue[] = Enums.PreservationJob.RecurrenceEnumTranslate;

  listWeekDay: KeyValue[] = JobHelper.getWeekDays();
  listMonth: KeyValue[] = JobHelper.getMonths();

  editValue: boolean = false;

  @ViewChild("hourFormatEditable") hourFormatEditableRef: ElementRef;

  get jobRecurrenceEnum(): typeof Enums.PreservationJob.RecurrenceEnum {
    return Enums.PreservationJob.RecurrenceEnum;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.enable]: ["", [SolidifyValidator]],
      [this.formDefinition.jobRecurrence]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.jobType]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.hour]: ["", [Validators.min(0), Validators.max(23), SolidifyValidator]],
      [this.formDefinition.weekDay]: ["", [SolidifyValidator]],
      [this.formDefinition.monthDay]: ["", [Validators.min(1), Validators.max(31), SolidifyValidator]],
      [this.formDefinition.month]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(preservationJob: PreservationJob): void {
    // Work around since this property has a validator with minimal value 1 but default value 0
    const scheduling: PreservationJobScheduling = ObjectUtil.clone(preservationJob.scheduling);
    if (scheduling.monthDay === 0) {
      scheduling.monthDay = undefined;
    }
    this.form = this._fb.group({
      [this.formDefinition.name]: [preservationJob.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.enable]: [preservationJob.enable, [SolidifyValidator]],
      [this.formDefinition.jobRecurrence]: [preservationJob.jobRecurrence, [Validators.required, SolidifyValidator]],
      [this.formDefinition.jobType]: [preservationJob.jobType, [Validators.required, SolidifyValidator]],
      [this.formDefinition.hour]: [this.getHours(preservationJob), [Validators.min(0), Validators.max(23), SolidifyValidator]],
      [this.formDefinition.weekDay]: [this.getWeekDay(preservationJob), [SolidifyValidator]],
      [this.formDefinition.monthDay]: [scheduling.monthDay, [Validators.min(1), Validators.max(31), SolidifyValidator]],
      [this.formDefinition.month]: [this.getMonth(preservationJob), [SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(preservationJob: PreservationJob): PreservationJob {
    const newScheduling: PreservationJobScheduling = {
      hour: this.form.get(this.formDefinition.hour).value === "" ? null : this.form.get(this.formDefinition.hour).value,
      month: this.form.get(this.formDefinition.month).value === "" ? null : this.form.get(this.formDefinition.month).value,
      monthDay: this.form.get(this.formDefinition.monthDay).value === "" ? null : this.form.get(this.formDefinition.monthDay).value,
      weekDay: this.form.get(this.formDefinition.weekDay).value === "" ? null : this.form.get(this.formDefinition.weekDay).value,
    };
    const newPreservationJob: PreservationJob = {
      name: preservationJob.name,
      enable: preservationJob.enable,
      jobRecurrence: preservationJob.jobRecurrence,
      jobType: preservationJob.jobType,
    };
    // set scheduling property only if any of the fields contains a value
    if (newScheduling.hour === null && newScheduling.month === null && newScheduling.weekDay === null && newScheduling.monthDay === null) {
      newPreservationJob.scheduling = null;
    } else {
      newPreservationJob.scheduling = newScheduling;
    }
    return newPreservationJob;
  }

  focusFieldEditable(): void {
    this.editValue = true;
    setTimeout(() => {
      this.hourFormatEditableRef.nativeElement.focus();
    }, 0);
  }

  checkHour(): void {
    this.editValue = false;
    const formControl = this.form.get(this.formDefinition.hour);
    if (formControl.invalid) {
      formControl.setValue("");
    }
  }

  private getHours(preservationJob: PreservationJob): number | undefined {
    return isNullOrUndefined(preservationJob.scheduling) ? undefined : preservationJob.scheduling.hour;
  }

  private getMonth(preservationJob: PreservationJob): string {
    if (isNullOrUndefined(preservationJob.scheduling) || isNullOrUndefined(preservationJob.scheduling.month)) {
      return undefined;
    }
    const month = JobHelper.getMonths().find(item => item.key === "" + preservationJob.scheduling.month);
    return month === undefined ? null : month.key;
  }

  private getWeekDay(preservationJob: PreservationJob): string {
    if (isNullOrUndefined(preservationJob.scheduling) || isNullOrUndefined(preservationJob.scheduling.weekDay)) {
      return undefined;
    }
    const weekDay = JobHelper.getWeekDays().find(item => item.key === "" + preservationJob.scheduling.weekDay);
    return weekDay === undefined ? null : weekDay.key;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() enable: string;
  @PropertyName() jobRecurrence: string;
  @PropertyName() jobType: string;
  @PropertyName() hour: string;
  @PropertyName() weekDay: string;
  @PropertyName() monthDay: string;
  @PropertyName() month: string;
}
