import {
  ChangeDetectorRef,
  Component,
  Inject,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {
  MAT_DIALOG_DATA,
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Store} from "@ngxs/store";
import {
  ErrorMessageWrapper,
  PreservationPlanningJobReportDetailErrorMessageDialog,
} from "@preservation-planning/job/components/dialogs/job-report-detail-error-message/preservation-planning-job-report-detail-error-message.dialog";
import {PreservationPlanningJobExecutionReportAction} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.action";
import {PreservationPlanningJobExecutionReportState} from "@preservation-planning/job/stores/job-execution/job-execution-report/preservation-planning-job-execution-report.state";
import {PreservationPlanningSipStatusHistoryAction} from "@preservation-planning/sip/stores/status-history/preservation-planning-sip-status-history.action";
import {PreservationPlanningSipStatusHistoryState} from "@preservation-planning/sip/stores/status-history/preservation-planning-sip-status-history.state";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {
  ArchivalStorageResourceApiEnum,
  IngestResourceApiEnum,
  PreIngestResourceApiEnum,
} from "@shared/enums/api.enum";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {SharedAipStatusHistoryAction} from "@shared/features/aip/stores/status-history/shared-aip-status-history.action";
import {SharedAipStatusHistoryState} from "@shared/features/aip/stores/status-history/shared-aip-status-history.state";
import {JobExecutionReportLine} from "@shared/models/business/job-execution-report-line.model";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {Observable} from "rxjs";
import {
  isNotNullNorUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-job-report-detail-dialog",
  templateUrl: "./preservation-planning-job-report-detail.dialog.html",
  styleUrls: ["./preservation-planning-job-report-detail.dialog.scss"],
})
export class PreservationPlanningJobReportDetailDialog extends SharedAbstractContainer implements OnInit, OnDestroy {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningJobExecutionReportState);
  listObs: Observable<JobExecutionReportLine[]> = MemoizedUtil.select(this._store, PreservationPlanningJobExecutionReportState, state => state.lineReport);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationPlanningJobExecutionReportState, state => state.queryParametersLines);

  historyAipObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, SharedAipStatusHistoryState, state => state.history);
  isLoadingHistoryAipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipStatusHistoryState);
  queryParametersHistoryAipObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, SharedAipStatusHistoryState, state => state.queryParameters);

  historySipObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, PreservationPlanningSipStatusHistoryState, state => state.history);
  isLoadingHistorySipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningSipStatusHistoryState);
  queryParametersHistorySipObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationPlanningSipStatusHistoryState, state => state.queryParameters);

  paramMessage: { resId: string } = {resId: StringUtil.stringEmpty};

  columns: DataTableColumns<JobExecutionReportLine>[];
  actions: DataTableActions<JobExecutionReportLine>[] = [
    {
      logo: IconNameEnum.error,
      callback: (jobExecutionReportLine: JobExecutionReportLine) => this.showMessageError(jobExecutionReportLine),
      placeholder: current => MARK_AS_TRANSLATABLE("crud.list.action.errorMessage"),
      displayOnCondition: (jobExecutionReportLine: JobExecutionReportLine) => jobExecutionReportLine.status !== Enums.PreservationJob.JobExecutionReportStatusEnum.PROCESSED,
    },
    {
      logo: IconNameEnum.history,
      callback: (jobExecutionReportLine: JobExecutionReportLine) => this.historyAip(jobExecutionReportLine),
      placeholder: current => LabelTranslateEnum.showHistoryStatus,
      displayOnCondition: (jobExecutionReportLine: JobExecutionReportLine) => jobExecutionReportLine.status !== Enums.PreservationJob.JobExecutionReportStatusEnum.ERROR,
    },
  ];

  constructor(private readonly _store: Store,
              private readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialogRef: MatDialogRef<PreservationPlanningJobReportDetailDialog>,
              protected readonly _dialog: MatDialog,
              @Inject(MAT_DIALOG_DATA) public data: ReportLinesWrapper) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.paramMessage = {resId: this.data.reportId};
    this.columns = [
      {
        field: "resId",
        header: LabelTranslateEnum.package,
        type: FieldTypeEnum.string,
        order: OrderEnum.descending,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
        translate: true,
        filterEnum: Enums.PreservationJob.ExecutionReportStatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "changeTime",
        header: LabelTranslateEnum.startDate,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
    ];
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this._store.dispatch(new PreservationPlanningJobExecutionReportAction.CleanQueryParametersLines());
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new PreservationPlanningJobExecutionReportAction.ChangeQueryParametersLines(queryParameters, this.data.executionId, this.data.preservationJobId, this.data.reportId));
    this._changeDetector.detectChanges();
  }

  showExecutionDetail(jobExecutionReportLine: JobExecutionReportLine): void {
    if (isNotNullNorUndefined(jobExecutionReportLine.url)) {
      if (jobExecutionReportLine.url.indexOf(IngestResourceApiEnum.sip) !== -1) {
        this._store.dispatch(new PreservationPlanningJobExecutionReportAction.GoToSip(jobExecutionReportLine.resId));
        this._dialogRef.close();
      } else if (jobExecutionReportLine.url.indexOf(PreIngestResourceApiEnum.deposits) !== -1) {
        this._store.dispatch(new PreservationPlanningJobExecutionReportAction.GoToDeposit(jobExecutionReportLine.resId));
        this._dialogRef.close();
      } else {
        for (const storage of ArchivalStorageResourceApiEnum.aipStorages) {
          if (jobExecutionReportLine.url.indexOf(storage.url) !== -1 || jobExecutionReportLine.url === ArchivalStorageResourceApiEnum.root) {
            this._store.dispatch(new PreservationPlanningJobExecutionReportAction.GoToAip(jobExecutionReportLine.resId));
            this._dialogRef.close();
            break;
          }
        }
      }
    }
  }

  showMessageError(jobExecutionReportLine: JobExecutionReportLine): void {
    this._dialog.open(PreservationPlanningJobReportDetailErrorMessageDialog, {
      data: {
        errorMessage: jobExecutionReportLine.errorMessage,
        resId: jobExecutionReportLine.resId,
      } as ErrorMessageWrapper,
    });
  }

  historyAip(jobExecutionReportLine: JobExecutionReportLine): void {
    //depending on the reportType, it will show the related aip or sip
    let dataForDialog: StatusHistoryDialog;
    if (this.data.jobType === Enums.PreservationJob.TypeEnum.PURGESUBMISSIONTEMPFILES || this.data.jobType === Enums.PreservationJob.TypeEnum.CLEANSUBMISSION) {
      dataForDialog = {
        parentId: null,
        resourceResId: jobExecutionReportLine.resId,
        name: LocalStateEnum.preservationPlanning_sip,
        statusHistory: this.historySipObs,
        isLoading: this.isLoadingHistorySipObs,
        queryParametersObs: this.queryParametersHistorySipObs,
        state: PreservationPlanningSipStatusHistoryAction,
      };
    } else {
      dataForDialog = {
        parentId: null,
        resourceResId: jobExecutionReportLine.resId,
        name: LocalStateEnum.shared_aip,
        statusHistory: this.historyAipObs,
        isLoading: this.isLoadingHistoryAipObs,
        queryParametersObs: this.queryParametersHistoryAipObs,
        state: SharedAipStatusHistoryAction,
      };
    }
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: dataForDialog,
    });
  }
}

export interface ReportLinesWrapper {
  linesReport: JobExecutionReportLine[];
  preservationJobId: string;
  executionId: string;
  reportId: string;
  jobType: Enums.PreservationJob.TypeEnum;
}
