import {
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {Enums} from "@enums";
import {PreservationJob} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {PreservationPlanningJobExecutionAction} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.action";
import {PreservationPlanningJobExecutionState} from "@preservation-planning/job/stores/job-execution/preservation-planning-job-execution.state";
import {PreservationPlanningJobAction} from "@preservation-planning/job/stores/preservation-planning-job.action";
import {PreservationPlanningJobState} from "@preservation-planning/job/stores/preservation-planning-job.state";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {JobExecution} from "@shared/models/business/job-execution.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  isNullOrUndefined,
  isTrue,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-job-execution-container",
  templateUrl: "./preservation-planning-job-execution-list.container.html",
  styleUrls: ["./preservation-planning-job-execution-list.container.scss"],
})
export class PreservationPlanningJobExecutionListContainer extends SharedAbstractPresentational implements OnInit {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningJobExecutionState);
  listObs: Observable<JobExecution[]> = MemoizedUtil.list(this._store, PreservationPlanningJobExecutionState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, PreservationPlanningJobExecutionState);
  jobObs: Observable<PreservationJob> = MemoizedUtil.current(this._store, PreservationPlanningJobState);
  columns: DataTableColumns<JobExecution>[];
  paramMessage: { name: string } = {name: StringUtil.stringEmpty};
  private readonly INTERVAL_REFRESH_IN_SECOND: number = 0.2;
  private readonly MAX_INTERVAL_REFRESH_IN_SECOND: number = 2;

  pollingToUpdateProgressBar: boolean = false;
  private _resId: string;

  constructor(private readonly _store: Store,
              private readonly _changeDetector: ChangeDetectorRef,
              private readonly _route: ActivatedRoute,
              private readonly _actions$: Actions,
              private readonly _dialog: MatDialog) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.defineColumns();
    this.retrieveCurrentModelWithUrl();

    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: this.INTERVAL_REFRESH_IN_SECOND,
      incrementInterval: true,
      maximumIntervalRefreshInSecond: this.MAX_INTERVAL_REFRESH_IN_SECOND,
      resetIntervalWhenUserMouseEvent: true,
      filter: () => isTrue(this.pollingToUpdateProgressBar),
      actionToDo: () => this.getAll(),
    }));

    this.subscribe(this.jobObs.pipe(filter(job => !isNullOrUndefined(job))), job => this.paramMessage = {name: job.name});
    this.subscribe(this.listObs.pipe(
      distinctUntilChanged(),
      filter(list => !isNullOrUndefined(list))), list => {
      this.pollingToUpdateProgressBar = list.findIndex(jobExec => jobExec.status === Enums.PreservationJob.ExecutionStatusEnum.IN_PROGRESS || jobExec.status === Enums.PreservationJob.ExecutionStatusEnum.READY) !== -1;
    });
    this._changeDetector.detectChanges();
  }

  protected retrieveResIdFromUrl(): void {
    this._resId = this._route.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    if (isNullOrUndefined(this._resId)) {
      this._resId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    }
  }

  protected retrieveCurrentModelWithUrl(): void {
    this.retrieveResIdFromUrl();
    this.getById(this._resId);
  }

  protected getById(id: string): void {
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "runNumber",
        header: LabelTranslateEnum.runNumber,
        type: FieldTypeEnum.string,
        order: OrderEnum.descending,
        isFilterable: false,
        isSortable: true,
        width: "80px",
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
        translate: true,
        filterEnum: Enums.PreservationJob.ExecutionStatusEnumTranslate,
        component: DataTableComponentEnum.status,
        width: "160px",
      },
      {
        field: "completionStatus",
        header: LabelTranslateEnum.completionStatus,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
        component: DataTableComponentEnum.jobExecutionProgression,
      },
      {
        field: "startDate",
        header: LabelTranslateEnum.startDate,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "endDate",
        header: LabelTranslateEnum.endDate,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
    ];
  }

  getAll(queryParameters?: QueryParameters): void {
    this._store.dispatch(new PreservationPlanningJobExecutionAction.GetAll(this._resId, queryParameters, true));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this.getAll(queryParameters);
    this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  start(): void {
    this._store.dispatch(new PreservationPlanningJobAction.VerifyBeforeStart(this._resId));
    this.subscribe(this._actions$.pipe(ofActionCompleted(PreservationPlanningJobAction.StartSuccess))
      .pipe(
        take(1),
        tap(result => {
          if (result.result.successful === true) {
            this.getAll();
          }
        })));
  }

  resume(): void {
    this.subscribe(this._actions$.pipe(ofActionCompleted(PreservationPlanningJobAction.ResumeSuccess))
      .pipe(
        take(1),
        tap(result => {
          if (result.result.successful === true) {
            this.getAll();
          }
        })));
    this._store.dispatch(new PreservationPlanningJobAction.Resume(this._resId));
  }

  showExecutionDetail(jobExecution: JobExecution): void {
    const pathExecution = RoutesEnum.preservationPlanning + AppRoutesEnum.separator + PreservationPlanningRoutesEnum.job + AppRoutesEnum.separator + PreservationPlanningRoutesEnum.jobDetail + AppRoutesEnum.separator + this._resId + AppRoutesEnum.separator + PreservationPlanningRoutesEnum.execution;
    const path = [pathExecution, jobExecution.resId];
    this._store.dispatch(new Navigate(path));
  }
}
