import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {Dip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  PreservationPlanningDipAipState,
  PreservationPlanningDipAipStateModel,
} from "@preservation-planning/dip/stores/aip/preservation-planning-dip-aip.state";
import {
  defaultPreservationPlanningDipDataFileValue,
  PreservationPlanningDipDataFileState,
  PreservationPlanningDipDataFileStateModel,
} from "@preservation-planning/dip/stores/data-file/preservation-planning-dip-data-file.state";
import {
  PreservationPlanningDipAction,
  preservationPlanningDipActionNameSpace,
} from "@preservation-planning/dip/stores/preservation-planning-dip.action";
import {
  PreservationPlanningDipStatusHistoryState,
  PreservationPlanningDipStatusHistoryStateModel,
} from "@preservation-planning/dip/stores/status-history/preservation-planning-dip-status-history.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {
  DepositRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultAssociationStateInitValue,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";

export interface PreservationPlanningDipStateModel extends ResourceStateModel<Dip> {
  [LocalStateEnum.preservationPlanning_dip_dataFile]: PreservationPlanningDipDataFileStateModel;
  isLoadingDataFile: boolean;
  [LocalStateEnum.preservationPlanning_dip_statusHistory]: PreservationPlanningDipStatusHistoryStateModel;
  [LocalStateEnum.preservationPlanning_dip_aip]: PreservationPlanningDipAipStateModel;
}

@Injectable()
@State<PreservationPlanningDipStateModel>({
  name: LocalStateEnum.preservationPlanning_dip,
  defaults: {
    ...defaultResourceStateInitValue(),
    [LocalStateEnum.preservationPlanning_dip_dataFile]: defaultPreservationPlanningDipDataFileValue(),
    isLoadingDataFile: false,
    [LocalStateEnum.preservationPlanning_dip_statusHistory]: {...defaultStatusHistoryInitValue()},
    [LocalStateEnum.preservationPlanning_dip_aip]: {...defaultAssociationStateInitValue()},
  },
  children: [
    PreservationPlanningDipDataFileState,
    PreservationPlanningDipStatusHistoryState,
    PreservationPlanningDipAipState,
  ],
})
export class PreservationPlanningDipState extends ResourceState<PreservationPlanningDipStateModel, Dip> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected httpClient: HttpClient,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningDipActionNameSpace,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.preservationPlanningDip,
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("dip.notification.resource.delete.success"),
      notificationResourceDeleteFailTextToTranslate: MARK_AS_TRANSLATABLE("dip.notification.resource.delete.fail"),
      notificationResourceUpdateFailTextToTranslate: MARK_AS_TRANSLATABLE("dip.notification.resource.update.fail"),
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.dip;
  }

  @Selector()
  static isLoading(state: PreservationPlanningDipStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: PreservationPlanningDipStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.info.name;
  }

  @Selector()
  static isLoadingWithDependency(state: PreservationPlanningDipStateModel): boolean {
    return this.isLoading(state)
      || StoreUtil.isLoadingState(state[LocalStateEnum.preservationPlanning_dip_aip]);
  }

  @Selector()
  static isReadyToBeDisplayed(state: PreservationPlanningDipStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current)
      && !isNullOrUndefined(state[LocalStateEnum.preservationPlanning_dip_aip].selected);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: PreservationPlanningDipStateModel): boolean {
    return true;
  }

  @OverrideDefaultAction()
  @Action(PreservationPlanningDipAction.UpdateSuccess)
  updateSuccess(ctx: StateContext<PreservationPlanningDipStateModel>, action: PreservationPlanningDipAction.UpdateSuccess): void {
    super.updateSuccess(ctx, action);

    ctx.dispatch([
      new Navigate([RoutesEnum.preservationPlanningDipDetail, action.model.resId, DepositRoutesEnum.data]),
    ]);
  }

  @Action(PreservationPlanningDipAction.Download)
  download(ctx: StateContext<PreservationPlanningDipStateModel>, action: PreservationPlanningDipAction.Download): void {
    const fileName = "dip_" + action.id + ".zip";
    this.downloadService.download(false, `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`, fileName);
  }

  @Action(PreservationPlanningDipAction.Resume)
  resume(ctx: StateContext<PreservationPlanningDipStateModel>, action: PreservationPlanningDipAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.httpClient.post<Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.RESUME}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new PreservationPlanningDipAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new PreservationPlanningDipAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new PreservationPlanningDipAction.ResumeFail(action));
          throw error;
        }),
      );
  }

  @Action(PreservationPlanningDipAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<PreservationPlanningDipStateModel>, action: PreservationPlanningDipAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.dip.action.resume.success"));
  }

  @Action(PreservationPlanningDipAction.ResumeFail)
  resumeFail(ctx: StateContext<PreservationPlanningDipStateModel>, action: PreservationPlanningDipAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.dip.action.resume.fail"));
  }
}
