import {
  Inject,
  Injectable,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {ApiActionEnum} from "@app/shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@app/shared/enums/api-resource-name.enum";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {DipDataFile} from "@preservation-planning/dip/models/dip-data-file.model";
import {
  PreservationPlanningDipDataFileAction,
  preservationPlanningDipDataFileActionNameSpace,
} from "@preservation-planning/dip/stores/data-file/preservation-planning-dip-data-file.action";
import {
  PreservationPlanningDipDataFileStatusHistoryState,
  PreservationPlanningDipDataFileStatusHistoryStateModel,
} from "@preservation-planning/dip/stores/data-file/status-history/preservation-planning-dip-data-file-status-history.state";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  NotificationService,
  SolidifyStateError,
} from "solidify-frontend";

export const defaultPreservationPlanningDipDataFileValue: () => PreservationPlanningDipDataFileStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.preservationPlanning_dip_dataFile_statusHistory]: defaultStatusHistoryInitValue(),
  });

export interface PreservationPlanningDipDataFileStateModel extends CompositionStateModel<DipDataFile> {
  [LocalStateEnum.preservationPlanning_dip_dataFile_statusHistory]: PreservationPlanningDipDataFileStatusHistoryStateModel;
}

@Injectable()
@State<PreservationPlanningDipDataFileStateModel>({
  name: LocalStateEnum.preservationPlanning_dip_dataFile,
  defaults: {
    ...defaultPreservationPlanningDipDataFileValue(),
  },
  children: [
    PreservationPlanningDipDataFileStatusHistoryState,
  ],
})
export class PreservationPlanningDipDataFileState extends CompositionState<PreservationPlanningDipDataFileStateModel, DipDataFile> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              @Inject(WINDOW) private _window: Window,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningDipDataFileActionNameSpace,
      resourceName: ApiResourceNameEnum.DATAFILE,
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.dip;
  }

  @Action(PreservationPlanningDipDataFileAction.Refresh)
  refresh(ctx: StateContext<PreservationPlanningDipDataFileStateModel>, action: PreservationPlanningDipDataFileAction.Refresh): void {
    ctx.dispatch(new PreservationPlanningDipDataFileAction.GetAll(action.parentId, undefined, true));
  }

  @Action(PreservationPlanningDipDataFileAction.Download)
  download(ctx: StateContext<PreservationPlanningDipDataFileStateModel>, action: PreservationPlanningDipDataFileAction.Download): void {
    const url = `${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.DL}`;
    this.downloadService.download(false, url, action.dataFile.fileName, action.dataFile.fileSize);
  }

  @Action(PreservationPlanningDipDataFileAction.Resume)
  resume(ctx: StateContext<PreservationPlanningDipDataFileStateModel>, action: PreservationPlanningDipDataFileAction.Resume): Observable<Result> {
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.RESUME}`)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            this.notificationService.showInformation(LabelTranslateEnum.resourceResumed);
            ctx.dispatch(new PreservationPlanningDipDataFileAction.GetAll(action.parentId, undefined, true));
          } else {
            this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          }
        }),
        catchError(error => {
          this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          throw new SolidifyStateError(this, error);
        }),
      );
  }
}
