import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {MatTabGroup} from "@angular/material/tabs";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Dip} from "@models";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {PreservationPlanningDipFormPresentational} from "@preservation-planning/dip/components/presentationals/dip-form/preservation-planning-dip-form.presentational";
import {PreservationPlanningDipAipAction} from "@preservation-planning/dip/stores/aip/preservation-planning-dip-aip.action";
import {
  PreservationPlanningDipAction,
  preservationPlanningDipActionNameSpace,
} from "@preservation-planning/dip/stores/preservation-planning-dip.action";
import {PreservationPlanningDipStateModel} from "@preservation-planning/dip/stores/preservation-planning-dip.state";
import {PreservationPlanningDipStatusHistoryAction} from "@preservation-planning/dip/stores/status-history/preservation-planning-dip-status-history.action";
import {PreservationPlanningDipStatusHistoryState} from "@preservation-planning/dip/stores/status-history/preservation-planning-dip-status-history.state";
import {Tab} from "@shared/components/containers/shared-tabs/shared-tabs.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  PreservationPlanningRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  isNullOrUndefined,
  MemoizedUtil,
  OverrideProperty,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-dip-detail-edit-routable",
  templateUrl: "./preservation-planning-dip-detail-edit.routable.html",
  styleUrls: ["./preservation-planning-dip-detail-edit.routable.scss"],
})
export class PreservationPlanningDipDetailEditRoutable extends SharedAbstractDetailEditRoutable<Dip, PreservationPlanningDipStateModel> implements OnInit, OnDestroy {
  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, PreservationPlanningDipStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, PreservationPlanningDipStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, PreservationPlanningDipStatusHistoryState, state => state.queryParameters);

  @ViewChild("formPresentational")
  readonly formPresentational: PreservationPlanningDipFormPresentational;

  @ViewChild("matTabGroup")
  readonly matTabGroup: MatTabGroup;

  message: string;

  @OverrideProperty()
  readonly editAvailable: boolean = false;

  @OverrideProperty()
  readonly deleteAvailable: boolean = false;

  readonly KEY_PARAM_NAME: keyof Dip & string = undefined;

  private currentTab: Tab;

  private get rootUrl(): string[] {
    return [RoutesEnum.preservationPlanningDipDetail, this._resId];
  }

  listTabs: Tab[] = [
    {
      id: TabEnum.METADATA,
      suffixUrl: PreservationPlanningRoutesEnum.dipMetadata,
      icon: IconNameEnum.metadata,
      titleToTranslate: LabelTranslateEnum.metadata,
      route: () => [...this.rootUrl, PreservationPlanningRoutesEnum.dipMetadata],
    },
    {
      id: TabEnum.FILES,
      suffixUrl: PreservationPlanningRoutesEnum.sipFiles,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.files,
      route: () => [...this.rootUrl, PreservationPlanningRoutesEnum.dipFiles],
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router,
              private readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.preservationPlanning_dip, preservationPlanningDipActionNameSpace, LocalStateEnum.preservationPlanning);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.retrieveCurrentModelWithUrl();
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new PreservationPlanningDipAipAction.GetAll(id));
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: PreservationPlanningDipStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

  listExtraButtons: ExtraButtonToolbar<Dip>[] = [
    {
      color: "primary",
      icon: IconNameEnum.download,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.COMPLETED,
      callback: () => this.download(),
      labelToTranslate: (current) => LabelTranslateEnum.download,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.resume,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.IN_ERROR,
      callback: () => this.resume(),
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.delete,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && this._securityService.isRoot(),
      callback: () => this.delete(),
      labelToTranslate: (current) => LabelTranslateEnum.delete,
      order: 40,
    },
  ];

  setCurrentTab($event: Tab): void {
    this.currentTab = $event;
  }

  download(): void {
    this._store.dispatch(new PreservationPlanningDipAction.Download(this._resId));
  }

  resume(): void {
    this._store.dispatch(new PreservationPlanningDipAction.Resume(this._resId));
  }

  delete(): void {
    this._store.dispatch(new PreservationPlanningDipAction.Delete(this._resId));
  }
}

enum TabEnum {
  METADATA = "METADATA",
  FILES = "FILES",
}
