import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {SharedAbstractFormPresentational} from "@app/shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@app/shared/models/base-form-definition.model";
import {Enums} from "@enums";
import {
  Aip,
  Dip,
} from "@models";
import {BaseResourceApiEnum} from "@shared/enums/api.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {KeyValueInfo} from "@shared/models/key-value-info.model";
import {Storage} from "@shared/models/storage.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {sharedOrganizationalUnitActionNameSpace} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrganizationalUnitState} from "@shared/stores/organizational-unit/shared-organizational-unit.state";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  DateUtil,
  EnumUtil,
  isNullOrUndefined,
  KeyValue,
  ObservableUtil,
  Override,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-preservation-planning-dip-form",
  templateUrl: "./preservation-planning-dip-form.presentational.html",
  styleUrls: ["./preservation-planning-dip-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PreservationPlanningDipFormPresentational extends SharedAbstractFormPresentational<Dip> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  accessEnumValues: KeyValue[] = Enums.Deposit.AccessEnumTranslate;
  dataSensitivityEnumValues: KeyValue[] = Enums.Deposit.DataSensitivityEnumTranslate;
  packageStatusEnumValuesTranslate: KeyValue[] = Enums.Package.StatusEnumTranslate;

  sharedOrgUnitActionNameSpace: ResourceNameSpace = sharedOrganizationalUnitActionNameSpace;
  sharedOrgUnitState: typeof SharedOrganizationalUnitState = SharedOrganizationalUnitState;

  @Input()
  isReady: boolean = false;

  @Input()
  listAip: Aip[];

  listStoragions: Storage[] = BaseResourceApiEnum.archivalStorageList;

  get packageStatusEnum(): typeof Enums.Package.StatusEnum {
    return Enums.Package.StatusEnum;
  }

  getSensitivityTooltip(dataSensitivity: Enums.Deposit.DataSensitivityEnum): string {
    return (EnumUtil.getKeyValue(Enums.Deposit.DataSensitivityEnumTranslate, dataSensitivity) as KeyValueInfo)?.infoToTranslate;
  }

  private readonly _orgUnitValueBS: BehaviorSubject<string | undefined> = new BehaviorSubject<string | undefined>(undefined);
  @Output("orgUnitChange")
  readonly orgUnitValueObs: Observable<string | undefined> = ObservableUtil.asObservable(this._orgUnitValueBS);

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  protected initNewForm(): void {
  }

  protected bindFormTo(dip: Dip): void {
    this.form = this._fb.group({
      [this.formDefinition.organizationalUnitId]: [dip.info.organizationalUnitId, [SolidifyValidator]],
      [this.formDefinition.resId]: [dip.resId, [SolidifyValidator]],
      [this.formDefinition.name]: [dip.info.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [dip.info.description, [SolidifyValidator]],
      [this.formDefinition.status]: [dip.info.status, [SolidifyValidator]],
      [this.formDefinition.access]: [dip.info.access, [SolidifyValidator]],
      [this.formDefinition.dataSensitivity]: [dip.info.dataSensitivity, [SolidifyValidator]],
      [this.formDefinition.complianceLevel]: [dip.info.complianceLevel, [SolidifyValidator]],
      [this.formDefinition.embargoAccess]: [dip.info.embargo?.access, []],
      [this.formDefinition.embargoNumberMonths]: [dip.info.embargo?.months, []],
      [this.formDefinition.embargoStartDate]: [isNullOrUndefined(dip.info.embargo?.startDate) ? "" : DateUtil.convertDateToDateTimeString(new Date(dip.info.embargo?.startDate))],
    });
    this.isValidWhenDisable = this.form.valid;
  }

  protected treatmentBeforeSubmit(dip: Dip): Dip {
    const dipToSubmit: Dip = {
      info: {
        name: this.form.get(this.formDefinition.name).value,
        description: this.form.get(this.formDefinition.description).value,
      },
    };
    return dipToSubmit;
  }

  @Override()
  protected disableSpecificField(): void {
    this.form.get(this.formDefinition.resId).disable();
    this.form.get(this.formDefinition.status).disable();
    this.form.get(this.formDefinition.access).disable();
    this.form.get(this.formDefinition.complianceLevel).disable();
  }

  goToAip(storagion: Storage, aipId: string): void {
    this._navigateBS.next([AppRoutesEnum.preservationPlanning, PreservationPlanningRoutesEnum.aip, String(storagion.index), SharedAipRoutesEnum.aipDetail, aipId]);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() organizationalUnitId: string;
  @PropertyName() resId: string;
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() status: string;
  @PropertyName() access: string;
  @PropertyName() dataSensitivity: string;
  @PropertyName() complianceLevel: string;
  @PropertyName() aip: string;
  @PropertyName() embargoAccess: string;
  @PropertyName() embargoNumberMonths: string;
  @PropertyName() embargoStartDate: string;
}
