import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {
  PreIngestResourceApiEnum,
} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {Deposit} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {DownloadService} from "@shared/services/download.service";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";
import {
  PreservationPlanningDepositAction,
  preservationPlanningDepositActionNameSpace,
} from "@preservation-planning/deposit/stores/preservation-planning-deposit.action";
import {ApiActionEnum} from "@shared/enums/api-action.enum";

export interface PreservationPlanningDepositStateModel extends ResourceStateModel<Deposit> {
  isLoadingDataFile: boolean;
}

@Injectable()
@State<PreservationPlanningDepositStateModel>({
  name: LocalStateEnum.preservationPlanning_deposit,
  defaults: {
    ...defaultResourceStateInitValue(),
    isLoadingDataFile: false,
  },
})
export class PreservationPlanningDepositState extends ResourceState<PreservationPlanningDepositStateModel, Deposit> {
  private readonly _DEPOSIT_FILE_DOWNLOAD_PREFIX: string = "deposit_";
  private readonly _ZIP_EXTENSION: string = ".zip";
  private readonly _FOLDER_QUERY_PARAM: string = "folder";

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected httpClient: HttpClient,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationPlanningDepositActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return PreIngestResourceApiEnum.deposits;
  }

  @Selector()
  static isLoading(state: PreservationPlanningDepositStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: PreservationPlanningDepositStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.title;
  }

  @Selector()
  static isLoadingWithDependency(state: PreservationPlanningDepositStateModel): boolean {
    return this.isLoading(state);
  }

  @Action(PreservationPlanningDepositAction.Download)
  download(ctx: StateContext<PreservationPlanningDepositStateModel>, action: PreservationPlanningDepositAction.Download): void {
    let fileName = this._DEPOSIT_FILE_DOWNLOAD_PREFIX + action.id;
    let url = `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`;
    if (isNotNullNorUndefined(action.fullFolderName)) {
      url = url + `?${this._FOLDER_QUERY_PARAM}=${action.fullFolderName}`;
      fileName = fileName + action.fullFolderName;
    }
    this.downloadService.download(false, url, fileName + this._ZIP_EXTENSION);
  }


}
