import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {DepositStateModel} from "@deposit/stores/deposit.state";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  PreservationAipStatusAction,
  preservationAipStatusActionNameSpace,
} from "@preservation-planning/aip-status/stores/preservation-planning-aip-status.action";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {PreservationPlanningResourceApiEnum} from "@shared/enums/api.enum";
import {AipCopyList} from "@shared/models/business/aip-copy-list.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  urlSeparator,
} from "solidify-frontend";

export interface PreservationPlanningAipStatusStateModel extends ResourceStateModel<AipCopyList> {
}

@Injectable()
@State<PreservationPlanningAipStatusStateModel>({
  name: LocalStateEnum.preservationPlanning_aipStatus,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
})
export class PreservationPlanningAipStatusState extends ResourceState<PreservationPlanningAipStatusStateModel, AipCopyList> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected httpClient: HttpClient) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: preservationAipStatusActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return PreservationPlanningResourceApiEnum.preservationAipStatuses;
  }

  @Action(PreservationAipStatusAction.DeleteCache)
  deleteCache(ctx: StateContext<DepositStateModel>, action: PreservationAipStatusAction.DeleteCache): Observable<string> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.post(this._urlResource + urlSeparator + ApiActionEnum.DELETE_CACHE, null, {
      responseType: "text",
    }).pipe(
      tap(result => {
        ctx.dispatch(new PreservationAipStatusAction.DeleteCacheSuccess(action));
      }),
      catchError(error => {
        ctx.dispatch(new PreservationAipStatusAction.DeleteCacheFail(action));
        throw new SolidifyStateError(this, error);
      }),
    );
  }

  @Action(PreservationAipStatusAction.DeleteCacheSuccess)
  deleteCacheSuccess(ctx: StateContext<DepositStateModel>, action: PreservationAipStatusAction.DeleteCacheSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("admin.preservation.aipStatuses.notification.deleteCache.success"));
  }

  @Action(PreservationAipStatusAction.DeleteCacheFail)
  deleteCacheFail(ctx: StateContext<DepositStateModel>, action: PreservationAipStatusAction.DeleteCacheFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("admin.preservation.aipStatuses.notification.deleteCache.fail"));
  }
}
