import {Injectable} from "@angular/core";
import {Enums} from "@enums";
import {Order} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  OrderMyOrderAipState,
  OrderMyOrderAipStateModel,
} from "@order/features/my-order/stores/aip/order-my-order-aip.state";
import {
  OrderMyOrderDipState,
  OrderMyOrderDipStateModel,
} from "@order/features/my-order/stores/dip/order-my-order-dip.state";
import {
  OrderMyOrderAction,
  orderMyOrderActionNameSpace,
} from "@order/features/my-order/stores/order-my-order.action";
import {
  OrderMyOrderStatusHistoryState,
  OrderMyOrderStatusHistoryStateModel,
} from "@order/features/my-order/stores/status-history/order-my-order-status-history.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {OrderArchive} from "@shared/models/business/order-archive.model";
import {Result} from "@shared/models/business/result.model";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultAssociationStateInitValue,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface OrderMyOrderStateModel extends ResourceStateModel<Order> {
  [LocalStateEnum.order_myOrder_statusHistory]: OrderMyOrderStatusHistoryStateModel;
  [LocalStateEnum.order_myOrder_dip]: OrderMyOrderDipStateModel | undefined;
  [LocalStateEnum.order_myOrder_aip]: OrderMyOrderAipStateModel | undefined;
}

@Injectable()
@State<OrderMyOrderStateModel>({
  name: LocalStateEnum.order_myOrder,
  defaults: {
    ...defaultResourceStateInitValue(),
    [LocalStateEnum.order_myOrder_statusHistory]: {...defaultStatusHistoryInitValue()},
    [LocalStateEnum.order_myOrder_dip]: {...defaultAssociationStateInitValue()},
    [LocalStateEnum.order_myOrder_aip]: {...defaultAssociationStateInitValue()},
  },
  children: [
    OrderMyOrderStatusHistoryState,
    OrderMyOrderDipState,
    OrderMyOrderAipState,
  ],
})
export class OrderMyOrderState extends ResourceState<OrderMyOrderStateModel, Order> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: orderMyOrderActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.orderMyOrderDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.orderMyOrderDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.orderMyOrder,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.myOrder.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.myOrder.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.myOrder.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.orders;
  }

  @Selector()
  static isLoading(state: OrderMyOrderStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: OrderMyOrderStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isLoadingWithDependency(state: OrderMyOrderStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: OrderMyOrderStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: OrderMyOrderStateModel): boolean {
    return true;
  }

  @Action(OrderMyOrderAction.Submit)
  submit(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.Submit): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(`${this._urlResource}/${action.resId}/${ApiActionEnum.SUBMIT}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new OrderMyOrderAction.SubmitSuccess(action));
          } else {
            ctx.dispatch(new OrderMyOrderAction.SubmitFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new OrderMyOrderAction.SubmitFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderMyOrderAction.SubmitSuccess)
  submitSuccess(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.SubmitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new OrderMyOrderAction.GetById((action.parentAction as OrderMyOrderAction.Submit).resId));
  }

  @Action(OrderMyOrderAction.SubmitFail)
  submitFail(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.SubmitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(OrderMyOrderAction.Resume)
  resume(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(`${this._urlResource}/${action.resId}/${ApiActionEnum.RESUME}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new OrderMyOrderAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new OrderMyOrderAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new OrderMyOrderAction.ResumeFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderMyOrderAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new OrderMyOrderAction.GetById((action.parentAction as OrderMyOrderAction.Resume).resId));
  }

  @Action(OrderMyOrderAction.ResumeFail)
  resumeFail(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(OrderMyOrderAction.Save)
  save(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.Save): Observable<OrderArchive> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<OrderArchive>(`${this._urlResource}/${action.resId}/${ApiActionEnum.SAVE}`)
      .pipe(
        tap(() => {
          ctx.dispatch(new OrderMyOrderAction.SaveSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new OrderMyOrderAction.SaveFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderMyOrderAction.SaveSuccess)
  saveSuccess(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.SaveSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(OrderMyOrderAction.SaveFail)
  saveFail(ctx: StateContext<OrderMyOrderStateModel>, action: OrderMyOrderAction.SaveFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
