import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {Enums} from "@enums";
import {Order} from "@models";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {
  OrderMyOrderAction,
  orderMyOrderActionNameSpace,
} from "@order/features/my-order/stores/order-my-order.action";
import {
  OrderMyOrderState,
  OrderMyOrderStateModel,
} from "@order/features/my-order/stores/order-my-order.state";
import {SharedAbstractListRoutable} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {RouterExtService} from "@shared/services/router-ext.service";
import {SecurityService} from "@shared/services/security.service";
import {
  MappingObjectUtil,
  MemoizedUtil,
  ObjectUtil,
  OrderEnum,
  Override,
  OverrideProperty,
  QueryParameters,
  QueryParametersUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-order-my-order-list-routable",
  templateUrl: "../../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.html",
  styleUrls: ["../../../../../../../shared/components/routables/shared-abstract-list/shared-abstract-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class OrderMyOrderListRoutable extends SharedAbstractListRoutable<Order, OrderMyOrderStateModel> implements OnInit {
  readonly KEY_CREATE_BUTTON: string = undefined;
  readonly KEY_BACK_BUTTON: string = LabelTranslateEnum.backToArchiveOrders;
  readonly KEY_PARAM_NAME: keyof Order & string = "name";
  private readonly KEY_CREATOR: string = "creation.who";
  private readonly KEY_PUBLIC_ORDER: string = "publicOrder";

  columnsSkippedToClear: string[] = [this.KEY_CREATOR, this.KEY_PUBLIC_ORDER];
  protected onlyMyOrders: boolean = false;

  @OverrideProperty()
  listNewId: string[] = SessionStorageHelper.getListItem(SessionStorageEnum.newOrderAvailable);

  constructor(protected readonly _store: Store,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              protected readonly _securityService: SecurityService) {
    super(_store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.order_myOrder, orderMyOrderActionNameSpace, {
      canCreate: false,
      canGoBack: _securityService.isRootOrAdmin(),
      listExtraButtons: [
        {
          color: "primary",
          icon: null,
          labelToTranslate: (current) => LabelTranslateEnum.showOnlyMyOrders,
          callback: (model, buttonElementRef, checked) => this.showOnlyMyOrder(checked),
          order: 40,
          isToggleButton: true,
          isToggleChecked: false,
        },
      ],
    }, LocalStateEnum.order);
  }

  showOnlyMyOrder(checked: boolean): void {
    let queryParameter = MemoizedUtil.queryParametersSnapshot(this._store, OrderMyOrderState);
    queryParameter = QueryParametersUtil.clone(queryParameter);
    if (checked) {
      this.updateQueryParameterWithCurrentUser(queryParameter, true);
      this.onlyMyOrders = true;
    } else {
      this.updateQueryParameterWithCurrentUser(queryParameter, false);
      this.onlyMyOrders = false;
    }
    this._store.dispatch(new OrderMyOrderAction.ChangeQueryParameters(queryParameter, true));
    this._changeDetector.detectChanges();
  }

  @Override()
  onQueryParametersEvent(queryParameters: QueryParameters): void {
    if (this.onlyMyOrders) {
      this.updateQueryParameterWithCurrentUser(queryParameters, true);
      this._store.dispatch(new OrderMyOrderAction.ChangeQueryParameters(queryParameters, true));
    } else {
      this.updateQueryParameterWithCurrentUser(queryParameters, false);
      this._store.dispatch(new OrderMyOrderAction.ChangeQueryParameters(queryParameters, true));
    }
    this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  conditionDisplayEditButton(model: Order | undefined): boolean {
    return model.status !== Enums.Order.StatusEnum.READY;
  }

  conditionDisplayDeleteButton(model: Order | undefined): boolean {
    return true;
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "queryType",
        header: LabelTranslateEnum.queryType,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.Order.QueryTypeEnumTranslate,
      },
      {
        field: "name",
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.Order.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "lastUpdate.when" as any,
        header: LabelTranslateEnum.updated,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: true,
        isSortable: true,
      },
    ];
  }

  private updateQueryParameterWithCurrentUser(queryParameters: QueryParameters, myOrder: boolean): QueryParameters | undefined {
    queryParameters = ObjectUtil.clone(queryParameters);
    queryParameters.search = ObjectUtil.clone(queryParameters.search);

    if (myOrder) {
      const userTargetedUid = this._store.selectSnapshot(AppState.currentUser).externalUid;
      MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_CREATOR, userTargetedUid);
      MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_PUBLIC_ORDER);
    } else {
      MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_PUBLIC_ORDER, "true");
      MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_CREATOR);
    }
    return queryParameters;
  }

}
