import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {AppCartDipAction} from "@app/stores/cart/dip/app-cart-dip.action";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Archive} from "@home/models/archive.model";
import {
  Dip,
  Order,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {OrderMyOrderAipAction} from "@order/features/my-order/stores/aip/order-my-order-aip.action";
import {OrderMyOrderAipState} from "@order/features/my-order/stores/aip/order-my-order-aip.state";
import {OrderMyOrderDipAction} from "@order/features/my-order/stores/dip/order-my-order-dip.action";
import {OrderMyOrderDipState} from "@order/features/my-order/stores/dip/order-my-order-dip.state";
import {
  OrderMyOrderAction,
  orderMyOrderActionNameSpace,
} from "@order/features/my-order/stores/order-my-order.action";
import {
  OrderMyOrderState,
  OrderMyOrderStateModel,
} from "@order/features/my-order/stores/order-my-order.state";
import {OrderMyOrderStatusHistoryAction} from "@order/features/my-order/stores/status-history/order-my-order-status-history.action";
import {OrderMyOrderStatusHistoryState} from "@order/features/my-order/stores/status-history/order-my-order-status-history.state";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditCommonRoutable} from "@shared/components/routables/shared-abstract-detail-edit-common/shared-abstract-detail-edit-common.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {SecurityService} from "@shared/services/security.service";
import {SharedArchiveAction} from "@shared/stores/archive/shared-archive.action";
import {sharedOrderActionNameSpace} from "@shared/stores/order/shared-order.action";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  map,
  take,
} from "rxjs/operators";
import {
  AssociationState,
  DateUtil,
  EnumUtil,
  isNonEmptyArray,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MemoizedUtil,
  OverrideProperty,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
} from "solidify-frontend";

@Component({
  selector: "dlcm-order-my-order-detail-edit-routable",
  templateUrl: "./order-my-order-detail-edit.routable.html",
  styleUrls: ["./order-my-order-detail-edit.routable.scss"],
})
export class OrderMyOrderDetailEditRoutable extends SharedAbstractDetailEditCommonRoutable<Order, OrderMyOrderStateModel> implements OnInit, OnDestroy {
  @Select(OrderMyOrderState.isLoadingWithDependency) isLoadingWithDependencyObs: Observable<boolean>;
  @Select(OrderMyOrderState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;

  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, OrderMyOrderStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderMyOrderStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, OrderMyOrderStatusHistoryState, state => state.queryParameters);

  listDipObs: Observable<Dip[]> = AssociationState.selected(this._store, OrderMyOrderDipState);
  listArchiveObs: Observable<Archive[]> = AssociationState.selected(this._store, OrderMyOrderAipState).pipe(
    distinctUntilChanged(),
    filter(listAip => isNonEmptyArray(listAip)),
    map(listAip => {
      const listArchive: Archive[] = [];
      listAip.forEach(aip => {
        listArchive.push({
          resId: aip.resId,
          organizationalUnitId: aip.info.organizationalUnitId,
          size: aip.archiveSize,
          accessLevel: aip.info.currentAccess as Enums.Deposit.AccessEnum,
          accessLevelToTranslate: EnumUtil.getLabel(Enums.Deposit.AccessEnumTranslate, aip.info.currentAccess),
          dataSensitivity: aip.info.dataSensitivity as Enums.Deposit.DataSensitivityEnum,
          dataSensitivityToTranslate: EnumUtil.getLabel(Enums.Deposit.DataSensitivityEnumTranslate, aip.info.dataSensitivity),
          contributors: [],
          files: String(aip.dataFileNumber),
          yearPublicationDate: DateUtil.extractYearFromDate(aip.creation.when),
          sizeDisplay: aip.smartSize,
          description: aip.info.description,
          title: aip.info.name,
          archiveMetadata: undefined,
          withThumbnail: true,
          isCollection: true,
        });
      });
      return listArchive;
    }),
  );
  isLoadingDipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderMyOrderDipState);
  isLoadingAipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderMyOrderAipState);

  @OverrideProperty()
  checkAvailableResourceNameSpace: ResourceNameSpace = sharedOrderActionNameSpace;

  readonly KEY_PARAM_NAME: keyof Order & string = "name";

  @OverrideProperty()
  readonly editAvailable: boolean = false;

  listExtraButtons: ExtraButtonToolbar<Order>[] = [
    {
      color: "primary",
      icon: IconNameEnum.resume,
      displayCondition: current => !this.isEdit && !isNullOrUndefined(current) && current.status === Enums.Order.StatusEnum.INERROR,
      callback: () => this.resume(),
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      order: 40,
    },
  ];

  private readonly INTERVAL_REFRESH_INITIAL_IN_SECOND: number = 5;
  private readonly INTERVAL_REFRESH_MAX_IN_SECOND: number = 60;

  ngOnInit(): void {
    super.ngOnInit();

    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: this.INTERVAL_REFRESH_INITIAL_IN_SECOND,
      incrementInterval: true,
      resetIntervalWhenUserMouseEvent: true,
      maximumIntervalRefreshInSecond: this.INTERVAL_REFRESH_MAX_IN_SECOND,
      continueUntil: () => isNullOrUndefined(this.current) || (this.current.status !== Enums.Order.StatusEnum.READY && this.current.status !== Enums.Order.StatusEnum.INERROR),
      actionToDo: () => this._store.dispatch(ResourceActionHelper.getById(this._resourceNameSpace, this._resId, true)),
    }));

    this.subscribe(this.currentObs.pipe(
      distinctUntilChanged(),
      filter(current => !isNullOrUndefined(current) && (this.current.status === Enums.Order.StatusEnum.READY || this.current.status === Enums.Order.StatusEnum.INERROR)),
      take(1),
      ),
      (current) => {
        this._store.dispatch(new OrderMyOrderDipAction.GetAll(this._resId));
        SessionStorageHelper.removeItemInList(SessionStorageEnum.newOrderAvailable, this._resId);
        this._store.dispatch(new OrderMyOrderAipAction.GetAll(this._resId)); // ALLOW TO UPDATE SIZE
      });

    this.subscribe(this.currentObs.pipe(
      distinctUntilChanged(),
      filter(current => !isNullOrUndefined(current)),
      take(1),
      ),
      (current) => {
        this._store.dispatch(new OrderMyOrderAipAction.GetAll(this._resId));
      });

    this.subscribe(PollingHelper.startPollingObs({
      // Manage case when we are on the page that become ready
      initialIntervalRefreshInSecond: 1,
      incrementInterval: true,
      resetIntervalWhenUserMouseEvent: true,
      filter: () => (isNotNullNorUndefined(this.current) && (this.current.status === Enums.Order.StatusEnum.READY || this.current.status === Enums.Order.StatusEnum.INERROR)) && SessionStorageHelper.getListItem(SessionStorageEnum.newOrderAvailable)
        .includes(this._resId),
      actionToDo: () => {
        SessionStorageHelper.removeItemInList(SessionStorageEnum.newOrderAvailable, this._resId);
      },
    }));
  }

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router,
              private readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.order_myOrder, orderMyOrderActionNameSpace, LocalStateEnum.order);
  }

  get enumUtil(): typeof EnumUtil {
    return EnumUtil;
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
  }

  getSubResourceWithParentId(id: string): void {
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: OrderMyOrderStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

  private resume(): void {
    this._store.dispatch(new OrderMyOrderAction.Resume(this._resId));
  }

  downloadDip(dip: Dip): void {
    this._store.dispatch(new AppCartDipAction.Download(dip.resId));
  }

  navigateToArchive(archive: Archive): void {
    this._store.dispatch(new Navigate([RoutesEnum.homeDetail, archive.resId]));
  }

  downloadArchive(archive: Archive): void {
    this._store.dispatch(new SharedArchiveAction.Download(archive));
  }

  downloadOrder(): void {
    this.subscribe(this.listDipObs.pipe(
      distinctUntilChanged(),
      filter(listDip => isNonEmptyArray(listDip)),
      take(1),
    ), listDip => {
      listDip.forEach(dip => this.downloadDip(dip));
    });
  }

  save(order: Order): void {
    this.update({
      model: order,
    });
  }
}
