import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {Enums} from "@enums";
import {Archive} from "@home/models/archive.model";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {Order} from "@models";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {SharedArchiveTileMode} from "@shared/components/presentationals/shared-archive-tile/shared-archive-tile.presentational";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  FormValidationHelper,
  isNullOrUndefined,
  ObservableUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-order-my-order",
  templateUrl: "./order-my-order.presentational.html",
  styleUrls: ["./order-my-order.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class OrderMyOrderPresentational extends SharedAbstractPresentational implements OnInit {
  _listArchives: Archive[];

  trackByFn(index: number, archive: Archive): string {
    return archive.resId;
  }

  private readonly _submitBS: BehaviorSubject<Order | undefined> = new BehaviorSubject<Order | undefined>(undefined);
  @Output("submit")
  readonly submitObs: Observable<Order | undefined> = ObservableUtil.asObservable(this._submitBS);

  private readonly _deleteOrderBS: BehaviorSubject<Order | undefined> = new BehaviorSubject<Order | undefined>(undefined);
  @Output("deleteOrder")
  readonly deleteOrderObs: Observable<Order | undefined> = ObservableUtil.asObservable(this._deleteOrderBS);

  private readonly _deleteAllArchiveBS: BehaviorSubject<void | undefined> = new BehaviorSubject<void | undefined>(undefined);
  @Output("deleteAllArchive")
  readonly deleteAllArchiveObs: Observable<void | undefined> = ObservableUtil.asObservable(this._deleteAllArchiveBS);

  private readonly _deleteArchiveBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("deleteArchive")
  readonly deleteArchiveObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._deleteArchiveBS);

  private readonly _navigateToArchiveBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("navigateToArchive")
  readonly navigateToArchiveObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._navigateToArchiveBS);

  private readonly _requestAccessBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("requestAccess")
  readonly requestAccessObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._requestAccessBS);

  private readonly _downloadArchiveBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("downloadArchive")
  readonly downloadArchiveObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._downloadArchiveBS);

  private readonly _downloadAllBS: BehaviorSubject<Order | undefined> = new BehaviorSubject<Order | undefined>(undefined);
  @Output("downloadAll")
  readonly downloadAllObs: Observable<Order | undefined> = ObservableUtil.asObservable(this._downloadAllBS);

  @Input()
  set listArchives(value: Archive[]) {
    this._listArchives = value;
    this.totalSize = 0;
    if (isNullOrUndefined(this._listArchives)) {
      return;
    }
    this._listArchives.forEach(a => this.totalSize += a.size);
    this.isGeneralAccessControlled = this._listArchives.some(archive => this.archiveAccessRightService.isAccessControlled(archive));
  }

  get listArchives(): Archive[] {
    return this._listArchives;
  }

  get orderStatusEnum(): typeof Enums.Order.StatusEnum {
    return Enums.Order.StatusEnum;
  }

  isGeneralAccessControlled: boolean = false;

  _order: Order;

  @Input()
  set order(value: Order) {
    this._order = value;
    if (isNullOrUndefined(this.order)) {
      return;
    }
    if (this.mode === "order") {
      if (this.order.status === Enums.Order.StatusEnum.READY) {
        this.iconOrderStatus = IconNameEnum.orderReady;
        this.messageOrderStatus = LabelTranslateEnum.orderReady;
        this.classOrderStatus = "is-ready";
      } else if (this.order.status === Enums.Order.StatusEnum.INERROR) {
        this.iconOrderStatus = IconNameEnum.orderInError;
        this.messageOrderStatus = LabelTranslateEnum.orderInError;
        this.classOrderStatus = "is-in-error";
      } else {
        this.iconOrderStatus = IconNameEnum.orderInProgress;
        this.messageOrderStatus = LabelTranslateEnum.orderInProgress;
        this.classOrderStatus = "is-in-progress";
      }
    }
  }

  get order(): Order {
    return this._order;
  }

  @Input()
  mode: SharedArchiveTileMode;

  form: FormGroup;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  totalSize: number = 0;
  iconOrderStatus: IconNameEnum | undefined = undefined;
  messageOrderStatus: string | undefined = undefined;
  classOrderStatus: "is-in-error" | "is-in-progress" | "is-ready" | undefined = undefined;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly archiveAccessRightService: ArchiveAccessRightService,
              public readonly breakpointService: BreakpointService) {
    super();
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  ngOnInit(): void {
    super.ngOnInit();
    if (isNullOrUndefined(this.order)) {
      this.initNewForm();
    } else {
      this.bindFormTo(this.order);
    }
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  protected bindFormTo(order: Order): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [order.name, [Validators.required, SolidifyValidator]],
    });
    this.form.controls[this.formDefinition.name].disable();
  }

  protected treatmentBeforeSubmit(order: Order): Order {
    return order;
  }

  submit(): void {
    this._submitBS.next({
      resId: isNullOrUndefined(this.order) ? undefined : this.order.resId,
      name: this.form.get(this.formDefinition.name).value,
    });
  }

  deleteAll(): void {
    this._deleteAllArchiveBS.next();
  }

  navigateToArchive(archive: Archive): void {
    this._navigateToArchiveBS.next(archive);
  }

  deleteArchive(archive: Archive): void {
    this._deleteArchiveBS.next(archive);
  }

  downloadArchive(archive: Archive): void {
    this._downloadArchiveBS.next(archive);
  }

  requestAccess(archive: Archive): void {
    this._requestAccessBS.next(archive);
  }

  deleteOrder(): void {
    this._deleteOrderBS.next(this.order);
  }

  downloadOrder(): void {
    this._downloadAllBS.next(this.order);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
}
