import {Injectable} from "@angular/core";
import {Enums} from "@enums";
import {Order} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  OrderAllOrderAipState,
  OrderAllOrderAipStateModel,
} from "@order/features/all-order/stores/aip/order-all-order-aip.state";
import {
  OrderAllOrderDipState,
  OrderAllOrderDipStateModel,
} from "@order/features/all-order/stores/dip/order-all-order-dip.state";
import {
  OrderAllOrderAction,
  orderAllOrderActionNameSpace,
} from "@order/features/all-order/stores/order-all-order.action";
import {
  OrderAllOrderOrderArchiveState,
  OrderAllOrderOrderArchiveStateModel,
} from "@order/features/all-order/stores/order-archive/order-all-order-order-archive.state";
import {
  OrderAllOrderStatusHistoryState,
  OrderAllOrderStatusHistoryStateModel,
} from "@order/features/all-order/stores/status-history/order-all-order-status-history.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {OrderArchive} from "@shared/models/business/order-archive.model";
import {Result} from "@shared/models/business/result.model";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultAssociationStateInitValue,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";

export interface OrderAllOrderStateModel extends ResourceStateModel<Order> {
  order_allOrder_statusHistory: OrderAllOrderStatusHistoryStateModel;
  order_allOrder_aip: OrderAllOrderAipStateModel | undefined;
  order_allOrder_dip: OrderAllOrderDipStateModel | undefined;
  order_allOrder_orderArchive: OrderAllOrderOrderArchiveStateModel | undefined;
}

@Injectable()
@State<OrderAllOrderStateModel>({
  name: LocalStateEnum.order_allOrder,
  defaults: {
    ...defaultResourceStateInitValue(),
    order_allOrder_statusHistory: {...defaultStatusHistoryInitValue()},
    order_allOrder_aip: {...defaultAssociationStateInitValue()},
    order_allOrder_dip: {...defaultAssociationStateInitValue()},
    order_allOrder_orderArchive: {...defaultResourceStateInitValue()},
  },
  children: [
    OrderAllOrderStatusHistoryState,
    OrderAllOrderOrderArchiveState,
    OrderAllOrderAipState,
    OrderAllOrderDipState,
  ],
})
export class OrderAllOrderState extends ResourceState<OrderAllOrderStateModel, Order> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: orderAllOrderActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.orderAllOrderDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.orderAllOrderDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.orderAllOrder,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.allOrder.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.allOrder.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("order.allOrder.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.orders;
  }

  @Selector()
  static isLoading(state: OrderAllOrderStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: OrderAllOrderStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isLoadingWithDependency(state: OrderAllOrderStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: OrderAllOrderStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: OrderAllOrderStateModel): boolean {
    return true;
  }

  @Action(OrderAllOrderAction.Submit)
  submit(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.Submit): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(`${this._urlResource}/${action.resId}/${ApiActionEnum.SUBMIT}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new OrderAllOrderAction.SubmitSuccess(action));
          } else {
            ctx.dispatch(new OrderAllOrderAction.SubmitFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new OrderAllOrderAction.SubmitFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderAllOrderAction.SubmitSuccess)
  submitSuccess(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.SubmitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new OrderAllOrderAction.GetById((action.parentAction as OrderAllOrderAction.Submit).resId));

    SessionStorageHelper.addItemInList(SessionStorageEnum.orderPending, (action.parentAction as OrderAllOrderAction.Submit).resId);

  }

  @Action(OrderAllOrderAction.SubmitFail)
  submitFail(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.SubmitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(OrderAllOrderAction.Resume)
  resume(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<any, Result>(`${this._urlResource}/${action.resId}/${ApiActionEnum.RESUME}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new OrderAllOrderAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new OrderAllOrderAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new OrderAllOrderAction.ResumeFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderAllOrderAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new OrderAllOrderAction.GetById((action.parentAction as OrderAllOrderAction.Resume).resId));
  }

  @Action(OrderAllOrderAction.ResumeFail)
  resumeFail(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  // @Action(OrderAllOrderAction.View)
  // view(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.View): Observable<OrderArchive[]> {
  //   ctx.patchState({
  //     isLoadingCounter: ctx.getState().isLoadingCounter + 1,
  //   });
  //
  //   return this.apiService.getByIdInPath<OrderArchive[]>(`${this._urlResource}/${action.resId}/${ApiActionEnum.VIEW}`)
  //     .pipe(
  //       tap((result) => {
  //         ctx.dispatch(new OrderAllOrderAction.ViewSuccess(action, result));
  //       }),
  //       catchError(error => {
  //         ctx.dispatch(new OrderAllOrderAction.ViewFail(action));
  //         throw new SolidifyStateError(this, error);
  //       }),
  //     );
  // }
  //
  // @Action(OrderAllOrderAction.ViewSuccess)
  // viewSuccess(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.ViewSuccess): void {
  //   ctx.patchState({
  //     isLoadingCounter: ctx.getState().isLoadingCounter - 1,
  //     orderArchives: action.orderArchives,
  //   });
  // }
  //
  // @Action(OrderAllOrderAction.ViewFail)
  // viewFail(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.ViewFail): void {
  //   ctx.patchState({
  //     isLoadingCounter: ctx.getState().isLoadingCounter - 1,
  //   });
  // }

  @Action(OrderAllOrderAction.Save)
  save(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.Save): Observable<OrderArchive> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<OrderArchive>(`${this._urlResource}/${action.resId}/${ApiActionEnum.SAVE}`)
      .pipe(
        tap(() => {
          ctx.dispatch(new OrderAllOrderAction.SaveSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new OrderAllOrderAction.SaveFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(OrderAllOrderAction.SaveSuccess)
  saveSuccess(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.SaveSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(OrderAllOrderAction.SaveFail)
  saveFail(ctx: StateContext<OrderAllOrderStateModel>, action: OrderAllOrderAction.SaveFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
