import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  Aip,
  Dip,
  Order,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Select,
  Store,
} from "@ngxs/store";
import {OrderAllOrderFormPresentational} from "@order/features/all-order/components/presentationals/all-order-form/order-all-order-form.presentational";
import {OrderAllOrderAipAction} from "@order/features/all-order/stores/aip/order-all-order-aip.action";
import {OrderAllOrderAipState} from "@order/features/all-order/stores/aip/order-all-order-aip.state";
import {OrderAllOrderDipAction} from "@order/features/all-order/stores/dip/order-all-order-dip.action";
import {OrderAllOrderDipState} from "@order/features/all-order/stores/dip/order-all-order-dip.state";
import {
  OrderAllOrderAction,
  orderAllOrderActionNameSpace,
} from "@order/features/all-order/stores/order-all-order.action";
import {
  OrderAllOrderState,
  OrderAllOrderStateModel,
} from "@order/features/all-order/stores/order-all-order.state";
import {OrderAllOrderOrderArchiveAction} from "@order/features/all-order/stores/order-archive/order-all-order-order-archive.action";
import {OrderAllOrderOrderArchiveState} from "@order/features/all-order/stores/order-archive/order-all-order-order-archive.state";
import {OrderAllOrderStatusHistoryAction} from "@order/features/all-order/stores/status-history/order-all-order-status-history.action";
import {OrderAllOrderStatusHistoryState} from "@order/features/all-order/stores/status-history/order-all-order-status-history.state";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditCommonRoutable} from "@shared/components/routables/shared-abstract-detail-edit-common/shared-abstract-detail-edit-common.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {OrderArchive} from "@shared/models/business/order-archive.model";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {SecurityService} from "@shared/services/security.service";
import {sharedOrderActionNameSpace} from "@shared/stores/order/shared-order.action";
import {MetadataUtil} from "@shared/utils/metadata.util";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  AssociationState,
  EnumUtil,
  isNullOrUndefined,
  MemoizedUtil,
  OverrideProperty,
  Paging,
  QueryParameters,
  ResourceNameSpace,
} from "solidify-frontend";

@Component({
  selector: "dlcm-order-all-order-detail-edit-routable",
  templateUrl: "./order-all-order-detail-edit.routable.html",
  styleUrls: ["./order-all-order-detail-edit.routable.scss"],
})
export class OrderAllOrderDetailEditRoutable extends SharedAbstractDetailEditCommonRoutable<Order, OrderAllOrderStateModel> implements OnInit, OnDestroy {
  @Select(OrderAllOrderState.isLoadingWithDependency) isLoadingWithDependencyObs: Observable<boolean>;
  @Select(OrderAllOrderState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;

  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, OrderAllOrderStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderAllOrderStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, OrderAllOrderStatusHistoryState, state => state.queryParameters);

  listAipObs: Observable<Aip[]> = AssociationState.selected(this._store, OrderAllOrderAipState);
  isLoadingAipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderAllOrderAipState);
  listDipObs: Observable<Aip[]> = AssociationState.selected(this._store, OrderAllOrderDipState);
  isLoadingDipObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderAllOrderDipState);

  listOrderArchivesObs: Observable<OrderArchive[]> = MemoizedUtil.list(this._store, OrderAllOrderOrderArchiveState);
  isLoadingOrderArchivesObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, OrderAllOrderOrderArchiveState);

  queryParametersAipObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, OrderAllOrderAipState);
  queryParametersOrderArchivesObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, OrderAllOrderOrderArchiveState);
  queryParametersDipObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, OrderAllOrderDipState);

  @ViewChild("formPresentational")
  readonly formPresentational: OrderAllOrderFormPresentational;

  @OverrideProperty()
  checkAvailableResourceNameSpace: ResourceNameSpace = sharedOrderActionNameSpace;

  get enumUtil(): typeof EnumUtil {
    return EnumUtil;
  }

  readonly KEY_PARAM_NAME: keyof Order & string = "name";

  listExtraButtons: ExtraButtonToolbar<Order>[] = [
    {
      color: "primary",
      icon: IconNameEnum.done,
      displayCondition: current => !this.isEdit && !isNullOrUndefined(current) && current.status === Enums.Order.StatusEnum.INPROGRESS,
      callback: () => this.save(),
      labelToTranslate: (current) => LabelTranslateEnum.save,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.submit,
      displayCondition: current => !this.isEdit && !isNullOrUndefined(current) && current.status === Enums.Order.StatusEnum.INPROGRESS && current.aipNumber > 0,
      callback: () => this.submit(),
      labelToTranslate: (current) => LabelTranslateEnum.submit,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.resume,
      displayCondition: current => !this.isEdit && !isNullOrUndefined(current) && current.status === Enums.Order.StatusEnum.INERROR,
      callback: () => this.resume(),
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      order: 40,
    },
  ];

  get orderStatusEnum(): typeof Enums.Order.StatusEnum {
    return Enums.Order.StatusEnum;
  }

  get metadataUtil(): typeof MetadataUtil {
    return MetadataUtil;
  }

  get accessLevelEnumTranslate(): typeof Enums.Deposit.AccessEnumTranslate {
    return Enums.Deposit.AccessEnumTranslate;
  }

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router,
              private readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.order_allOrder, orderAllOrderActionNameSpace, LocalStateEnum.order);
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new OrderAllOrderAipAction.GetAll(id));
    this._store.dispatch(new OrderAllOrderDipAction.GetAll(id));
    this.view();
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: OrderAllOrderStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

  private submit(): void {
    this._store.dispatch(new OrderAllOrderAction.Submit(this._resId));
  }

  private resume(): void {
    this._store.dispatch(new OrderAllOrderAction.Resume(this._resId));
  }

  private view(): void {
    this._store.dispatch(new OrderAllOrderOrderArchiveAction.GetAll(this._resId));
  }

  private save(): void {
    this._store.dispatch(new OrderAllOrderAction.Save(this._resId));
    this.subscribe(this._actions$.pipe(ofActionCompleted(OrderAllOrderAction.SaveSuccess))
      .pipe(
        take(1),
        tap(result => {
          if (result.result.successful) {
            this._store.dispatch(new OrderAllOrderAipAction.GetAll(this._resId));
          }
        }),
      ));
  }

  goToDip(dip: Dip): void {
    this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningDipDetail, dip.resId]));
  }

  goToAipDownloaded(aip: Aip): void {
    // TODO When aip downloaded page done
    this.goToAip(aip.resId);
  }

  goToAip(aipId: string): void {
    const storagionNode = 1;
    this._store.dispatch(new Navigate([AppRoutesEnum.preservationPlanning, PreservationPlanningRoutesEnum.aip, storagionNode, SharedAipRoutesEnum.aipDetail, aipId]));
  }

  pageChangeMatchingArchives($event: Paging): void {
    const queryParameters = new QueryParameters();
    queryParameters.paging = $event;
    this._store.dispatch(new OrderAllOrderOrderArchiveAction.GetAll(this._resId, queryParameters));
  }

  pageChangeAipsSelected($event: Paging): void {
    const queryParameters = new QueryParameters();
    queryParameters.paging = $event;
    this._store.dispatch(new OrderAllOrderAipAction.GetAll(this._resId, queryParameters));
  }

  pageChangeDipsAvailable($event: Paging): void {
    const queryParameters = new QueryParameters();
    queryParameters.paging = $event;
    this._store.dispatch(new OrderAllOrderDipAction.GetAll(this._resId, queryParameters));
  }
}
